<?php
// Include initialization file
require_once 'init.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('auth/login.php');
}

// Initialize filter variables
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$limit = 20;
$offset = ($page - 1) * $limit;
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$userFilter = isset($_GET['user']) ? intval($_GET['user']) : 0;
$dateStart = isset($_GET['date_start']) ? $_GET['date_start'] : '';
$dateEnd = isset($_GET['date_end']) ? $_GET['date_end'] : '';

// Build query conditions
$conditions = [];
$params = [];

if (!empty($search)) {
    $conditions[] = "activity LIKE ?s";
    $params[] = "%$search%";
}

if ($userFilter > 0) {
    $conditions[] = "user_id = ?i";
    $params[] = $userFilter;
}

if (!empty($dateStart)) {
    $formattedDateStart = formatDateToMysql($dateStart) . ' 00:00:00';
    $conditions[] = "created_at >= ?s";
    $params[] = $formattedDateStart;
}

if (!empty($dateEnd)) {
    $formattedDateEnd = formatDateToMysql($dateEnd) . ' 23:59:59';
    $conditions[] = "created_at <= ?s";
    $params[] = $formattedDateEnd;
}

// Construct WHERE clause
$whereClause = '';
if (!empty($conditions)) {
    $whereClause = "WHERE " . implode(" AND ", $conditions);
}

// Count total records for pagination
$totalQuery = "SELECT COUNT(*) FROM activity_logs $whereClause";
$totalLogs = $db->getOne($totalQuery, ...$params);

// Calculate total pages
$totalPages = ceil($totalLogs / $limit);

// Get activities with pagination
$query = "
    SELECT al.*, u.name as user_name, u.role as user_role 
    FROM activity_logs al
    LEFT JOIN users u ON al.user_id = u.id
    $whereClause
    ORDER BY al.created_at DESC
    LIMIT ?i, ?i
";

// Add limit and offset to params
$params[] = $offset;
$params[] = $limit;

// Execute query
$activities = $db->getAll($query, ...$params);

// Get users for filter dropdown
$users = $db->getAll("SELECT id, name, role FROM users ORDER BY name ASC");
?>
<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta content="width=device-width, initial-scale=1, maximum-scale=1, shrink-to-fit=no" name="viewport">
    <title>Aktivitas Sistem &mdash; Sistem Rekam Medis Ortodontik</title>

    <!-- General CSS Files -->
    <link rel="stylesheet" href="assets/modules/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="assets/modules/fontawesome/css/all.min.css">

    <!-- CSS Libraries -->
    <link rel="stylesheet" href="assets/modules/bootstrap-daterangepicker/daterangepicker.css">
    <link rel="stylesheet" href="assets/modules/select2/dist/css/select2.min.css">
    <link rel="stylesheet" href="assets/modules/izitoast/css/iziToast.min.css">

    <!-- Template CSS -->
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/components.css">
</head>

<body>
    <div id="app">
        <div class="main-wrapper main-wrapper-1">
            <!-- Header -->
            <?php include 'layouts/header.php'; ?>

            <!-- Sidebar -->
            <?php include 'layouts/sidebar.php'; ?>

            <!-- Main Content -->
            <div class="main-content">
                <section class="section">
                    <div class="section-header">
                        <h1>Aktivitas Sistem</h1>
                        <div class="section-header-breadcrumb">
                            <div class="breadcrumb-item active"><a href="index.php">Dashboard</a></div>
                            <div class="breadcrumb-item">Aktivitas Sistem</div>
                        </div>
                    </div>

                    <div class="section-body">
                        <h2 class="section-title">Log Aktivitas Sistem</h2>
                        <p class="section-lead">
                            Halaman ini menampilkan semua aktivitas yang terjadi di dalam sistem.
                        </p>

                        <div class="row">
                            <div class="col-12">
                                <div class="card">
                                    <div class="card-header">
                                        <h4>Filter</h4>
                                        <div class="card-header-action">
                                            <a data-collapse="#filter-collapse" class="btn btn-icon btn-info" href="#"><i
                                                    class="fas fa-plus"></i></a>
                                        </div>
                                    </div>
                                    <div class="collapse" id="filter-collapse">
                                        <div class="card-body">
                                            <form action="" method="GET">
                                                <div class="row">
                                                    <div class="form-group col-md-4">
                                                        <label for="search">Kata Kunci</label>
                                                        <input type="text" class="form-control" id="search" name="search"
                                                            value="<?php echo htmlspecialchars($search); ?>"
                                                            placeholder="Cari aktivitas...">
                                                    </div>
                                                    <div class="form-group col-md-4">
                                                        <label for="user">Pengguna</label>
                                                        <select class="form-control select2" id="user" name="user">
                                                            <option value="0">Semua Pengguna</option>
                                                            <?php foreach ($users as $user): ?>
                                                            <option value="<?php echo $user['id']; ?>"
                                                                <?php echo $userFilter == $user['id'] ? 'selected' : ''; ?>>
                                                                <?php echo $user['name']; ?>
                                                                (<?php echo ucfirst($user['role']); ?>)
                                                            </option>
                                                            <?php endforeach; ?>
                                                        </select>
                                                    </div>
                                                    <div class="form-group col-md-4">
                                                        <label>Rentang Tanggal</label>
                                                        <div class="input-group">
                                                            <div class="input-group-prepend">
                                                                <div class="input-group-text">
                                                                    <i class="fas fa-calendar"></i>
                                                                </div>
                                                            </div>
                                                            <input type="text" class="form-control daterange-cus"
                                                                name="date_range"
                                                                value="<?php echo !empty($dateStart) && !empty($dateEnd) ? $dateStart . ' - ' . $dateEnd : ''; ?>">
                                                            <input type="hidden" name="date_start" id="date_start"
                                                                value="<?php echo $dateStart; ?>">
                                                            <input type="hidden" name="date_end" id="date_end"
                                                                value="<?php echo $dateEnd; ?>">
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="text-right">
                                                    <a href="activities.php" class="btn btn-secondary mr-2">Reset</a>
                                                    <button type="submit" class="btn btn-primary">Filter</button>
                                                </div>
                                            </form>
                                        </div>
                                    </div>
                                </div>

                                <div class="card">
                                    <div class="card-header">
                                        <h4>Daftar Aktivitas</h4>
                                        <div class="card-header-action">
                                            <span class="badge badge-primary">Total: <?php echo $totalLogs; ?></span>
                                        </div>
                                    </div>
                                    <div class="card-body p-0">
                                        <div class="table-responsive">
                                            <table class="table table-striped">
                                                <thead>
                                                    <tr>
                                                        <th>Waktu</th>
                                                        <th>Pengguna</th>
                                                        <th>Role</th>
                                                        <th>Aktivitas</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php if (count($activities) > 0): ?>
                                                    <?php foreach ($activities as $activity): ?>
                                                    <tr>
                                                        <td>
                                                            <?php 
                                                                echo date('d-m-Y H:i:s', strtotime($activity['created_at']));
                                                            ?>
                                                            <br>
                                                            <small class="text-muted">
                                                                <?php 
                                                                    $timestamp = strtotime($activity['created_at']);
                                                                    echo human_time_diff($timestamp, time());
                                                                ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <?php if ($activity['user_id']): ?>
                                                            <?php echo $activity['user_name']; ?>
                                                            <?php else: ?>
                                                            <span class="text-muted">Sistem</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <?php if ($activity['user_role']): ?>
                                                            <div
                                                                class="badge badge-<?php echo $activity['user_role'] === 'admin' ? 'danger' : ($activity['user_role'] === 'doctor' ? 'warning' : 'info'); ?>">
                                                                <?php echo ucfirst($activity['user_role']); ?>
                                                            </div>
                                                            <?php else: ?>
                                                            <div class="badge badge-dark">Sistem</div>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td><?php echo $activity['activity']; ?></td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                    <?php else: ?>
                                                    <tr>
                                                        <td colspan="4" class="text-center">Tidak ada data aktivitas yang
                                                            ditemukan</td>
                                                    </tr>
                                                    <?php endif; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                    <div class="card-footer text-center">
                                        <nav class="d-inline-block">
                                            <ul class="pagination mb-0">
                                                <?php if ($page > 1): ?>
                                                <li class="page-item">
                                                    <a class="page-link"
                                                        href="?page=<?php echo $page - 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $userFilter > 0 ? '&user=' . $userFilter : ''; ?><?php echo !empty($dateStart) ? '&date_start=' . urlencode($dateStart) : ''; ?><?php echo !empty($dateEnd) ? '&date_end=' . urlencode($dateEnd) : ''; ?>"
                                                        tabindex="-1"><i class="fas fa-chevron-left"></i></a>
                                                </li>
                                                <?php endif; ?>

                                                <?php
                                                $startPage = max(1, $page - 2);
                                                $endPage = min($totalPages, $page + 2);

                                                if ($startPage > 1) {
                                                    echo '<li class="page-item disabled"><a class="page-link" href="#">...</a></li>';
                                                }

                                                for ($i = $startPage; $i <= $endPage; $i++):
                                                ?>
                                                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                                    <a class="page-link"
                                                        href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $userFilter > 0 ? '&user=' . $userFilter : ''; ?><?php echo !empty($dateStart) ? '&date_start=' . urlencode($dateStart) : ''; ?><?php echo !empty($dateEnd) ? '&date_end=' . urlencode($dateEnd) : ''; ?>"><?php echo $i; ?></a>
                                                </li>
                                                <?php endfor; ?>

                                                <?php if ($endPage < $totalPages): ?>
                                                <li class="page-item disabled"><a class="page-link" href="#">...</a></li>
                                                <?php endif; ?>

                                                <?php if ($page < $totalPages): ?>
                                                <li class="page-item">
                                                    <a class="page-link"
                                                        href="?page=<?php echo $page + 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $userFilter > 0 ? '&user=' . $userFilter : ''; ?><?php echo !empty($dateStart) ? '&date_start=' . urlencode($dateStart) : ''; ?><?php echo !empty($dateEnd) ? '&date_end=' . urlencode($dateEnd) : ''; ?>"><i
                                                            class="fas fa-chevron-right"></i></a>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                        </nav>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            </div>

            <!-- Footer -->
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <!-- General JS Scripts -->
    <script src="assets/modules/jquery.min.js"></script>
    <script src="assets/modules/popper.js"></script>
    <script src="assets/modules/tooltip.js"></script>
    <script src="assets/modules/bootstrap/js/bootstrap.min.js"></script>
    <script src="assets/modules/nicescroll/jquery.nicescroll.min.js"></script>
    <script src="assets/modules/moment.min.js"></script>
    <script src="assets/js/stisla.js"></script>

    <!-- JS Libraies -->
    <script src="assets/modules/bootstrap-daterangepicker/daterangepicker.js"></script>
    <script src="assets/modules/select2/dist/js/select2.full.min.js"></script>

    <!-- Template JS File -->
    <script src="assets/js/scripts.js"></script>
    <script src="assets/js/custom.js"></script>

    <script>
    $(document).ready(function() {
        // Initialize select2
        $('.select2').select2();

        // Initialize daterangepicker
        $('.daterange-cus').daterangepicker({
            locale: {
                format: 'DD-MM-YYYY',
                applyLabel: 'Pilih',
                cancelLabel: 'Batal',
                daysOfWeek: ['Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab'],
                monthNames: ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus',
                    'September', 'Oktober', 'November', 'Desember'
                ],
                firstDay: 1
            },
            drops: 'down',
            opens: 'right',
            autoApply: true,
            maxDate: moment()
        }, function(start, end, label) {
            $('#date_start').val(start.format('DD-MM-YYYY'));
            $('#date_end').val(end.format('DD-MM-YYYY'));
        });

        // Open filter collapse if any filter is active
        <?php if (!empty($search) || $userFilter > 0 || !empty($dateStart) || !empty($dateEnd)): ?>
        $('#filter-collapse').collapse('show');
        <?php endif; ?>
    });
    </script>
</body>

</html> 