<?php
// Include initialization file
require_once '../../init.php';

// Check if user is logged in and has admin role
if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('Anda tidak memiliki izin untuk mengakses halaman ini.', 'danger');
    redirect('../../index.php');
}

// Check if user ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    setFlashMessage('ID pengguna tidak valid.', 'danger');
    redirect('index.php');
}

$userId = intval($_GET['id']);

// Get user data
$user = $db->getRow("SELECT * FROM users WHERE id = ?i", $userId);
if (!$user) {
    setFlashMessage('Pengguna tidak ditemukan.', 'danger');
    redirect('index.php');
}

// Initialize variables
$name = $user['name'];
$username = $user['username'];
$email = $user['email'];
$role = $user['role'];
$password = '';
$confirmPassword = '';
$errors = [];

// Process form submission
if (isPostRequest()) {
    // Get form data
    $name = trim($_POST['name']);
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $role = trim($_POST['role']);
    $password = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    
    // Validate input
    if (empty($name)) {
        $errors['name'] = 'Nama harus diisi.';
    }
    
    if (empty($username)) {
        $errors['username'] = 'Username harus diisi.';
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors['username'] = 'Username hanya boleh berisi huruf, angka, dan underscore.';
    } else {
        // Check if username already exists (excluding current user)
        $usernameExists = $db->getOne("SELECT COUNT(*) FROM users WHERE username = ?s AND id != ?i", $username, $userId);
        if ($usernameExists > 0) {
            $errors['username'] = 'Username sudah digunakan.';
        }
    }
    
    if (empty($email)) {
        $errors['email'] = 'Email harus diisi.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = 'Format email tidak valid.';
    } else {
        // Check if email already exists (excluding current user)
        $emailExists = $db->getOne("SELECT COUNT(*) FROM users WHERE email = ?s AND id != ?i", $email, $userId);
        if ($emailExists > 0) {
            $errors['email'] = 'Email sudah digunakan.';
        }
    }
    
    if (empty($role)) {
        $errors['role'] = 'Peran harus dipilih.';
    } elseif (!in_array($role, ['admin', 'doctor', 'staff'])) {
        $errors['role'] = 'Peran tidak valid.';
    }
    
    // Check if this is the last admin
    if ($user['role'] === 'admin' && $role !== 'admin') {
        $adminCount = $db->getOne("SELECT COUNT(*) FROM users WHERE role = 'admin'");
        if ($adminCount <= 1) {
            $errors['role'] = 'Tidak dapat mengubah peran admin terakhir.';
        }
    }
    
    // Password validation only if provided
    if (!empty($password)) {
        if (strlen($password) < 6) {
            $errors['password'] = 'Password minimal 6 karakter.';
        }
        
        if ($password !== $confirmPassword) {
            $errors['confirm_password'] = 'Konfirmasi password tidak cocok.';
        }
    }
    
    // If no errors, update user
    if (empty($errors)) {
        // Prepare update query
        $updateFields = [
            "name = ?s",
            "username = ?s",
            "email = ?s",
            "role = ?s",
            "updated_by = ?i",
            "updated_at = NOW()"
        ];
        
        $updateParams = [
            $name,
            $username,
            $email,
            $role,
            $_SESSION['user_id']
        ];
        
        // Add password update if provided
        if (!empty($password)) {
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $updateFields[] = "password = ?s";
            $updateParams[] = $hashedPassword;
        }
        
        $updateQuery = "UPDATE users SET " . implode(", ", $updateFields) . " WHERE id = ?i";
        $updateParams[] = $userId;
        
        // Execute update
        $result = $db->query($updateQuery, ...$updateParams);
        
        if ($result) {
            // Log activity
            logActivity("Mengubah data pengguna: $name ($username)");
            
            setFlashMessage('Data pengguna berhasil diperbarui.', 'success');
            redirect('index.php');
        } else {
            $errorMessage = 'Terjadi kesalahan saat memperbarui data pengguna.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta content="width=device-width, initial-scale=1, maximum-scale=1, shrink-to-fit=no" name="viewport">
    <title>Edit Pengguna &mdash; Sistem Rekam Medis Ortodontik</title>

    <!-- General CSS Files -->
    <link rel="stylesheet" href="../../assets/modules/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="../../assets/modules/fontawesome/css/all.min.css">

    <!-- CSS Libraries -->
    <link rel="stylesheet" href="../../assets/modules/select2/dist/css/select2.min.css">
    <link rel="stylesheet" href="../../assets/modules/izitoast/css/iziToast.min.css">

    <!-- Template CSS -->
    <link rel="stylesheet" href="../../assets/css/style.css">
    <link rel="stylesheet" href="../../assets/css/components.css">
</head>

<body>
    <div id="app">
        <div class="main-wrapper main-wrapper-1">
            <!-- Header -->
            <?php include '../../layouts/header.php'; ?>

            <!-- Sidebar -->
            <?php include '../../layouts/sidebar.php'; ?>

            <!-- Main Content -->
            <div class="main-content">
                <section class="section">
                    <div class="section-header">
                        <h1>Edit Pengguna</h1>
                        <div class="section-header-breadcrumb">
                            <div class="breadcrumb-item active"><a href="../../index.php">Dashboard</a></div>
                            <div class="breadcrumb-item"><a href="index.php">Manajemen Pengguna</a></div>
                            <div class="breadcrumb-item">Edit Pengguna</div>
                        </div>
                    </div>

                    <div class="section-body">
                        <div class="row">
                            <div class="col-12">
                                <div class="card">
                                    <div class="card-header">
                                        <h4>Form Edit Pengguna</h4>
                                    </div>
                                    <div class="card-body">
                                        <?php if (isset($errorMessage)): ?>
                                        <div class="alert alert-danger"><?php echo $errorMessage; ?></div>
                                        <?php endif; ?>

                                        <form action="" method="POST">
                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Nama Lengkap <span class="text-danger">*</span></label>
                                                <div class="col-sm-12 col-md-7">
                                                    <input type="text" class="form-control <?php echo isset($errors['name']) ? 'is-invalid' : ''; ?>" name="name" value="<?php echo htmlspecialchars($name); ?>" required>
                                                    <?php if (isset($errors['name'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['name']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Username <span class="text-danger">*</span></label>
                                                <div class="col-sm-12 col-md-7">
                                                    <input type="text" class="form-control <?php echo isset($errors['username']) ? 'is-invalid' : ''; ?>" name="username" value="<?php echo htmlspecialchars($username); ?>" required>
                                                    <?php if (isset($errors['username'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['username']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                    <small class="form-text text-muted">Username hanya boleh berisi huruf, angka, dan underscore.</small>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Email <span class="text-danger">*</span></label>
                                                <div class="col-sm-12 col-md-7">
                                                    <input type="email" class="form-control <?php echo isset($errors['email']) ? 'is-invalid' : ''; ?>" name="email" value="<?php echo htmlspecialchars($email); ?>" required>
                                                    <?php if (isset($errors['email'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['email']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Peran <span class="text-danger">*</span></label>
                                                <div class="col-sm-12 col-md-7">
                                                    <select class="form-control select2 <?php echo isset($errors['role']) ? 'is-invalid' : ''; ?>" name="role" required>
                                                        <option value="">Pilih Peran</option>
                                                        <option value="admin" <?php echo $role === 'admin' ? 'selected' : ''; ?>>Admin</option>
                                                        <option value="doctor" <?php echo $role === 'doctor' ? 'selected' : ''; ?>>Dokter</option>
                                                        <option value="staff" <?php echo $role === 'staff' ? 'selected' : ''; ?>>Staff</option>
                                                    </select>
                                                    <?php if (isset($errors['role'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['role']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Password</label>
                                                <div class="col-sm-12 col-md-7">
                                                    <input type="password" class="form-control <?php echo isset($errors['password']) ? 'is-invalid' : ''; ?>" name="password">
                                                    <?php if (isset($errors['password'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['password']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                    <small class="form-text text-muted">Biarkan kosong jika tidak ingin mengubah password.</small>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3">Konfirmasi Password</label>
                                                <div class="col-sm-12 col-md-7">
                                                    <input type="password" class="form-control <?php echo isset($errors['confirm_password']) ? 'is-invalid' : ''; ?>" name="confirm_password">
                                                    <?php if (isset($errors['confirm_password'])): ?>
                                                    <div class="invalid-feedback">
                                                        <?php echo $errors['confirm_password']; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>

                                            <div class="form-group row mb-4">
                                                <label class="col-form-label text-md-right col-12 col-md-3 col-lg-3"></label>
                                                <div class="col-sm-12 col-md-7">
                                                    <button type="submit" class="btn btn-primary">Simpan</button>
                                                    <a href="index.php" class="btn btn-secondary">Batal</a>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            </div>

            <!-- Footer -->
            <?php include '../../layouts/footer.php'; ?>
        </div>
    </div>

    <!-- General JS Scripts -->
    <script src="../../assets/modules/jquery.min.js"></script>
    <script src="../../assets/modules/popper.js"></script>
    <script src="../../assets/modules/tooltip.js"></script>
    <script src="../../assets/modules/bootstrap/js/bootstrap.min.js"></script>
    <script src="../../assets/modules/nicescroll/jquery.nicescroll.min.js"></script>
    <script src="../../assets/modules/moment.min.js"></script>
    <script src="../../assets/js/stisla.js"></script>

    <!-- JS Libraries -->
    <script src="../../assets/modules/select2/dist/js/select2.full.min.js"></script>

    <!-- Template JS File -->
    <script src="../../assets/js/scripts.js"></script>
    <script src="../../assets/js/custom.js"></script>

    <script>
    $(document).ready(function() {
        $('.select2').select2();
    });
    </script>
</body>

</html> 