<?php
/**
 * Helper functions for the application
 */

/**
 * Sanitize input
 * @param string $input Input to sanitize
 * @return string Sanitized input
 */
function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Check if user is logged in
 * @return bool True if user is logged in, false otherwise
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

/**
 * Check if user has admin role
 * @return bool True if user is admin, false otherwise
 */
function isAdmin() {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

/**
 * Check if user has doctor role
 * @return bool True if user is doctor, false otherwise
 */
function isDoctor() {
    return isset($_SESSION['user_role']) && ($_SESSION['user_role'] === 'doctor' || $_SESSION['user_role'] === 'admin');
}

/**
 * Redirect to a page
 * @param string $page Page to redirect to
 */
function redirect($page) {
    header("Location: $page");
    exit;
}

/**
 * Get current page name
 * @return string Current page name
 */
function getCurrentPage() {
    return basename($_SERVER['PHP_SELF']);
}

/**
 * Display alert message
 * @param string $message Message to display
 * @param string $type Type of alert (success, danger, warning, info)
 * @return string HTML alert
 */
function alert($message, $type = 'info') {
    return '<div class="alert alert-' . $type . ' alert-dismissible fade show" role="alert">
        ' . $message . '
        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
            <span aria-hidden="true">&times;</span>
        </button>
    </div>';
}

/**
 * Set flash message to be displayed on next page load
 * @param string $message Message to display
 * @param string $type Type of alert (success, danger, warning, info)
 */
function setFlashMessage($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}

/**
 * Display flash message if exists and clear it
 * @return string HTML alert or empty string
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'];
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        return alert($message, $type);
    }
    return '';
}

/**
 * Calculate age from birth date
 * @param string $birthDate Birth date in Y-m-d format
 * @return int Age in years
 */
function calculateAge($birthDate) {
    $birth = new DateTime($birthDate);
    $now = new DateTime();
    $interval = $now->diff($birth);
    return $interval->y;
}

/**
 * Format date to Indonesian format
 * @param string $date Date in Y-m-d format
 * @return string Date in d-m-Y format
 */
function formatDate($date) {
    if (empty($date)) return '';
    return date('d-m-Y', strtotime($date));
}

/**
 * Format date to MySQL format
 * @param string $date Date in d-m-Y format
 * @return string Date in Y-m-d format
 */
function formatDateToMysql($date) {
    if (empty($date)) return '';
    return date('Y-m-d', strtotime($date));
}

/**
 * Check if a string is a valid date
 * @param string $date Date to check
 * @param string $format Date format
 * @return bool True if date is valid, false otherwise
 */
function isValidDate($date, $format = 'Y-m-d') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

/**
 * Generate a random string
 * @param int $length Length of the random string
 * @return string Random string
 */
function generateRandomString($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $randomString;
}

/**
 * Check if the current request is a POST request
 * @return bool True if request is POST, false otherwise
 */
function isPostRequest() {
    return $_SERVER['REQUEST_METHOD'] === 'POST';
}

/**
 * Upload a file
 * @param array $file File from $_FILES
 * @param string $targetDir Target directory
 * @param array $allowedTypes Allowed file types
 * @param int $maxSize Maximum file size in bytes
 * @return array Result of upload with keys 'success', 'message', and 'filepath' if successful
 */
function uploadFile($file, $targetDir, $allowedTypes = [], $maxSize = 5242880) {
    // Check if file was uploaded without errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'Ukuran file melebihi batas maksimum yang diizinkan oleh server.',
            UPLOAD_ERR_FORM_SIZE => 'Ukuran file melebihi batas maksimum yang diizinkan oleh form.',
            UPLOAD_ERR_PARTIAL => 'File hanya terupload sebagian.',
            UPLOAD_ERR_NO_FILE => 'Tidak ada file yang diunggah.',
            UPLOAD_ERR_NO_TMP_DIR => 'Folder temporary tidak ditemukan.',
            UPLOAD_ERR_CANT_WRITE => 'Gagal menyimpan file.',
            UPLOAD_ERR_EXTENSION => 'Upload file dihentikan oleh extension.'
        ];
        
        $errorMessage = isset($errorMessages[$file['error']]) 
            ? $errorMessages[$file['error']] 
            : 'Terjadi kesalahan saat upload file.';
        
        return ['success' => false, 'message' => $errorMessage];
    }
    
    // Check file size
    if ($file['size'] > $maxSize) {
        return ['success' => false, 'message' => 'Ukuran file terlalu besar.'];
    }
    
    // Check file type
    $fileType = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!empty($allowedTypes) && !in_array($fileType, $allowedTypes)) {
        return [
            'success' => false, 
            'message' => 'Tipe file tidak diizinkan. Tipe file yang diizinkan: ' . implode(', ', $allowedTypes)
        ];
    }
    
    // Create target directory if it doesn't exist
    if (!is_dir($targetDir)) {
        mkdir($targetDir, 0755, true);
    }
    
    // Generate unique filename
    $fileName = md5(uniqid() . $file['name']) . '.' . $fileType;
    $targetFile = $targetDir . '/' . $fileName;
    
    // Move uploaded file to target directory
    if (move_uploaded_file($file['tmp_name'], $targetFile)) {
        return [
            'success' => true,
            'message' => 'File berhasil diunggah.',
            'filepath' => $targetFile
        ];
    } else {
        return ['success' => false, 'message' => 'Terjadi kesalahan saat memindahkan file.'];
    }
}

/**
 * Format a number with Indonesian number format
 * @param float $number Number to format
 * @param int $decimals Number of decimal places
 * @return string Formatted number
 */
function formatNumber($number, $decimals = 2) {
    return number_format($number, $decimals, ',', '.');
}

/**
 * Get the menu active state
 * @param string $pageName Current page name
 * @param string $menuItem Menu item to check
 * @return string Active state class if current page matches menu item
 */
function isMenuActive($pageName, $menuItem) {
    return strpos($pageName, $menuItem) !== false ? 'active' : '';
}

/**
 * Get required field marker
 * @return string HTML for required field marker
 */
function getRequiredFieldMarker() {
    return '<span class="text-danger">*</span>';
}

/**
 * Convert JSON to associative array
 * @param string $json JSON string
 * @return array Associative array
 */
function jsonToArray($json) {
    return json_decode($json, true) ?: [];
}

/**
 * Check if value is within the specified range
 * @param float $value Value to check
 * @param float $average Average value
 * @param float $range Range
 * @return bool True if value is within range, false otherwise
 */
function isWithinRange($value, $average, $range) {
    return ($value >= ($average - $range) && $value <= ($average + $range));
}

/**
 * Get cephalometric conclusion based on value and range
 * @param float $value Value to check
 * @param float $average Average value
 * @param float $range Range
 * @param array $conclusions Array of conclusions [below, normal, above]
 * @return string Conclusion
 */
function getCephalometricConclusion($value, $average, $range, $conclusions) {
    if ($value < ($average - $range)) {
        return $conclusions[0]; // Below
    } else if ($value > ($average + $range)) {
        return $conclusions[2]; // Above
    } else {
        return $conclusions[1]; // Normal
    }
}

function logActivity($activity) {
    global $db;
    $db->query("INSERT INTO activity_logs (user_id, activity) VALUES (?s, ?s)", $_SESSION['user_id'], $activity);
}

function formatDateTime($dateTime) {
    return date('d-m-Y H:i:s', strtotime($dateTime));
}