<?php
/**
 * @author nvb608 (nikolay.kachalov@gmail.com)
 * @link https://github.com/colshrapnel/safemysql
 * @license http://www.opensource.org/licenses/mit-license.php MIT
 * 
 * Safe and convenient way to handle SQL queries utilizing type-hinted placeholders.
 */

class SafeMySQL
{

    private $conn;
    private $stats;
    private $emode;
    private $exname;
    private $defaults = array(
        'host'      => 'localhost',
        'user'      => 'root',
        'pass'      => '',
        'db'        => 'test',
        'port'      => NULL,
        'socket'    => NULL,
        'pconnect'  => FALSE,
        'charset'   => 'utf8',
        'errmode'   => 'exception', //or 'error'
        'exception' => 'Exception', //Exception class name
    );

    const RESULT_ASSOC = MYSQLI_ASSOC;
    const RESULT_NUM   = MYSQLI_NUM;

    function __construct($opt = array())
    {
        $opt = array_merge($this->defaults, $opt);

        $this->emode  = $opt['errmode'];
        $this->exname = $opt['exception'];

        if ($opt['pconnect']) {
            $opt['host'] = "p:" . $opt['host'];
        }

        @$this->conn = mysqli_connect($opt['host'], $opt['user'], $opt['pass'], $opt['db'], $opt['port'], $opt['socket']);
        if (!$this->conn) {
            $this->error(mysqli_connect_errno() . " " . mysqli_connect_error());
        }

        mysqli_set_charset($this->conn, $opt['charset']) or $this->error(mysqli_error($this->conn));
        $this->stats = array();
    }

    /**
     * Conventional function to run a query with placeholders. A mysqli_query wrapper with placeholders support
     * 
     * Examples:
     * $db->query("DELETE FROM table WHERE id=?i", $id);
     * $db->query("INSERT INTO table SET foo=?s, bar=?s", $foo, $bar);
     * $db->query("SELECT * FROM table WHERE foo LIKE ?p", "%$foo%");
     *
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return resource|FALSE whatever mysqli_query returns
     */
    public function query()
    {
        return $this->rawQuery($this->prepareQuery(func_get_args()));
    }

    /**
     * Conventional function to fetch single row. 
     * 
     * @param resource $result - myqli result
     * @param int $mode - optional fetch mode, RESULT_ASSOC|RESULT_NUM, default RESULT_ASSOC
     * @return array|false whatever mysqli_fetch_array returns
     */
    public function fetch($result, $mode = self::RESULT_ASSOC)
    {
        return mysqli_fetch_array($result, $mode);
    }

    /**
     * Conventional function to get number of affected rows. 
     * 
     * @return int whatever mysqli_affected_rows returns
     */
    public function affectedRows()
    {
        return mysqli_affected_rows($this->conn);
    }

    /**
     * Conventional function to get last insert id
     * 
     * @return int whatever mysqli_insert_id returns
     */
    public function insertId()
    {
        return mysqli_insert_id($this->conn);
    }

    /**
     * Conventional function to get number of rows in the resultset. 
     * 
     * @param resource $result - myqli result
     * @return int whatever mysqli_num_rows returns
     */
    public function numRows($result)
    {
        return mysqli_num_rows($result);
    }

    /**
     * Helper function to get scalar value right out of query and possibly cached
     * 
     * Examples:
     * $name = $db->getOne("SELECT name FROM table WHERE id=1");
     * $name = $db->getOne("SELECT name FROM table WHERE id=?i", $id);
     *
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return string|FALSE either first column of the first row of resultset or FALSE if none found
     */
    public function getOne()
    {
        $query = $this->prepareQuery(func_get_args());
        if ($res = $this->rawQuery($query)) {
            $row = $this->fetch($res);
            if (is_array($row)) {
                return reset($row);
            }
            $this->free($res);
        }
        return FALSE;
    }

    /**
     * Helper function to get single row right out of query and possibly cached
     * 
     * Examples:
     * $data = $db->getRow("SELECT * FROM table WHERE id=1");
     * $data = $db->getOne("SELECT * FROM table WHERE id=?i", $id);
     *
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return array|FALSE either associative array contains first row of resultset or FALSE if none found
     */
    public function getRow()
    {
        $query = $this->prepareQuery(func_get_args());
        if ($res = $this->rawQuery($query)) {
            $ret = $this->fetch($res);
            $this->free($res);
            return $ret;
        }
        return FALSE;
    }

    /**
     * Helper function to get single column right out of query and possibly cached
     * 
     * Examples:
     * $ids = $db->getCol("SELECT id FROM table WHERE cat=1");
     * $ids = $db->getCol("SELECT id FROM tags WHERE tag=?s", $tag);
     *
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return array|FALSE either enumerated array of first fields of all rows of resultset or FALSE if none found
     */
    public function getCol()
    {
        $ret   = array();
        $query = $this->prepareQuery(func_get_args());
        if ($res = $this->rawQuery($query)) {
            while ($row = $this->fetch($res)) {
                $ret[] = reset($row);
            }
            $this->free($res);
        }
        return $ret;
    }

    /**
     * Helper function to get all the rows of resultset right out of query and possibly cached
     * 
     * Examples:
     * $data = $db->getAll("SELECT * FROM table");
     * $data = $db->getAll("SELECT * FROM table LIMIT ?i,?i", $start, $rows);
     *
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return array enumerated 2d array contains the resultset. Empty if no rows found.
     */
    public function getAll()
    {
        $ret   = array();
        $query = $this->prepareQuery(func_get_args());
        if ($res = $this->rawQuery($query)) {
            while ($row = $this->fetch($res)) {
                $ret[] = $row;
            }
            $this->free($res);
        }
        return $ret;
    }

    /**
     * Helper function to get all the rows of resultset into indexed array right out of query and possibly cached
     * 
     * Examples:
     * $data = $db->getInd("id", "SELECT * FROM table");
     * $data = $db->getInd("id", "SELECT * FROM table LIMIT ?i,?i", $start, $rows);
     *
     * @param string $index - name of the field which value is used as a key for indexed array
     * @param string $query - an SQL query with placeholders
     * @param mixed  $arg,... unlimited number of arguments to match placeholders in the query
     * @return array - associative 2d array contains the resultset. Empty if no rows found.
     */
    public function getInd()
    {
        $args  = func_get_args();
        $index = array_shift($args);
        $query = $this->prepareQuery($args);

        $ret = array();
        if ($res = $this->rawQuery($query)) {
            while ($row = $this->fetch($res)) {
                $ret[$row[$index]] = $row;
            }
            $this->free($res);
        }
        return $ret;
    }

    /**
     * Function to get last executed query. 
     * 
     * @return string|NULL either last executed query or NULL if were none
     */
    public function lastQuery()
    {
        $last = end($this->stats);
        return $last['query'];
    }

    /**
     * Function to get all query statistics. 
     * 
     * @return array contains all executed queries with timings and errors
     */
    public function getStats()
    {
        return $this->stats;
    }

    /**
     * Free the resultset
     */
    public function free($result)
    {
        mysqli_free_result($result);
    }

    /**
     * Helper function to execute prepared query with parameters
     */
    private function rawQuery($query)
    {
        $start = microtime(TRUE);
        $res   = mysqli_query($this->conn, $query);
        $timer = microtime(TRUE) - $start;

        $this->stats[] = array(
            'query' => $query,
            'time'  => $timer,
        );
        if (!$res) {
            $error = mysqli_error($this->conn);
            
            end($this->stats);
            $key = key($this->stats);
            $this->stats[$key]['error'] = $error;
            $this->error("$error. Full query: [$query]");
        }
        return $res;
    }

    /**
     * Function to handle mysqli errors
     */
    private function error($err)
    {
        if ($this->emode == 'error') {
            trigger_error($err);
            return;
        }
        throw new $this->exname($err);
    }

    /**
     * Function to parse placeholders either in the full query or a query part
     * placeholders in the original query must be either ?a, ?s, ?i, ?n, ?p, ?l, ?d, ?u with specific meaning
     * or ?{} numeric indexed ones will be replaced with the corresponding arguments in the list
     * 
     * ?s - string (quotes will be added)
     * ?i - integer
     * ?n - number (quotes will not be added but the string will be checked for being a valid numeric value)
     * ?a - array (will be processed with the IN operator, quotes will be added depending on the array content)
     * ?u - union (will be processed with the UNION operator, a semicolon must be used to separate the queries)
     * ?p - literally parsed (the placeholder will be replaced with the argument without any processing)
     * ?l - list (will be processed as a list of fields, separated by comma)
     * ?d - date (quotes will be added)
     * 
     * @param array $args - two dimensional array containing the original query splitted as an array
     *                      and arbitrary number of arguments
     * @return string - processed query to be passed to mysqli_query for execution
     */
    private function prepareQuery($args)
    {
        // Need to access the internal mysqli connection
        $conn = $this->conn;
        
        // This is the placeholder parsing function
        $parse = function($matches) use($args, $conn) {
            $key = $matches[1];
            
            // Check if the key is an integer?
            if (isset($args[$key])) {
                $arg = $args[$key];
            } else {
                // If not, check if it's a named placeholder
                $position = strpos($key, '_');
                if ($position !== false) {
                    $prefix = substr($key, 0, $position);
                    $key = substr($key, $position + 1);
                    if (isset($args[$key])) {
                        $arg = $args[$key];
                        if ($prefix == 's') return "'" . mysqli_real_escape_string($conn, $arg) . "'";
                        if ($prefix == 'i') return intval($arg);
                        if ($prefix == 'n') return is_numeric($arg) ? $arg : "'" . mysqli_real_escape_string($conn, $arg) . "'";
                        if ($prefix == 'a') {
                            $placeholders = array();
                            foreach ((array)$arg as $val) {
                                $placeholders[] = is_numeric($val) ? $val : "'" . mysqli_real_escape_string($conn, $val) . "'";
                            }
                            return implode(',', $placeholders);
                        }
                        if ($prefix == 'u') {
                            $queries = explode(';', $arg);
                            return implode(' UNION ', $queries);
                        }
                        if ($prefix == 'p') return $arg;
                        if ($prefix == 'l') {
                            $fields = array();
                            foreach ((array)$arg as $field) {
                                $fields[] = $field;
                            }
                            return implode(',', $fields);
                        }
                        if ($prefix == 'd') return "'" . mysqli_real_escape_string($conn, $arg) . "'";
                    }
                }
                return $matches[0]; // If no match, return the original placeholder
            }
            
            // Handle different placeholder types
            if ($key[0] == 's') return "'" . mysqli_real_escape_string($conn, $arg) . "'";
            if ($key[0] == 'i') return intval($arg);
            if ($key[0] == 'n') return is_numeric($arg) ? $arg : "'" . mysqli_real_escape_string($conn, $arg) . "'";
            if ($key[0] == 'a') {
                $placeholders = array();
                foreach ((array)$arg as $val) {
                    $placeholders[] = is_numeric($val) ? $val : "'" . mysqli_real_escape_string($conn, $val) . "'";
                }
                return implode(',', $placeholders);
            }
            if ($key[0] == 'u') {
                $queries = explode(';', $arg);
                return implode(' UNION ', $queries);
            }
            if ($key[0] == 'p') return $arg;
            if ($key[0] == 'l') {
                $fields = array();
                foreach ((array)$arg as $field) {
                    $fields[] = $field;
                }
                return implode(',', $fields);
            }
            if ($key[0] == 'd') return "'" . mysqli_real_escape_string($conn, $arg) . "'";
            
            return $matches[0]; // If no match, return the original placeholder
        };
        
        // Get the query and other arguments
        $query = array_shift($args);
        
        // Replace the placeholders in the query
        $result = preg_replace_callback('~\?([a-zA-Z0-9_]+)~', $parse, $query);
        
        return $result;
    }
} 