<?php
// Include initialization file
require_once '../init.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}

// Check if user is a doctor (only doctors can edit medical records)
if (!isDoctor()) {
    setFlashMessage('Anda tidak memiliki izin untuk mengedit rekam medis.', 'danger');
    redirect('../index.php');
}

// Check if ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    setFlashMessage('ID rekam medis tidak valid.', 'danger');
    redirect('index.php');
}

$recordId = intval($_GET['id']);

// Get medical record data
$record = $db->getRow("SELECT * FROM medical_records WHERE id = ?i", $recordId);

if (!$record) {
    setFlashMessage('Rekam medis tidak ditemukan.', 'danger');
    redirect('index.php');
}

// Check if record is complete - if yes, only view is allowed
if ($record['is_complete'] == 1) {
    setFlashMessage('Rekam medis sudah selesai dan tidak dapat diedit.', 'info');
    redirect('view.php?id=' . $recordId);
}

// Get patient data
$patient = $db->getRow("SELECT 
    *,
    TIMESTAMPDIFF(YEAR, birth_date, CURDATE()) as age
    FROM patients 
    WHERE id = ?i", $record['patient_id']);

if (!$patient) {
    setFlashMessage('Data pasien tidak ditemukan.', 'danger');
    redirect('index.php');
}

// Initialize variables
$errorMessage = '';
$successMessage = '';
$currentSection = isset($_GET['section']) ? intval($_GET['section']) : 1;

// Get extra-oral examination data if it exists
$extraOral = $db->getRow("SELECT * FROM extra_oral_examinations WHERE medical_record_id = ?i", $recordId);

// Get intra-oral examination data if it exists
$intraOral = $db->getRow("SELECT * FROM intra_oral_examinations WHERE medical_record_id = ?i", $recordId);

// Parse bad habits from stored format to array
$badHabits = [];
$badHabitsOther = '';
if (!empty($record['bad_habits'])) {
    $badHabitsArray = explode(', ', $record['bad_habits']);
    foreach ($badHabitsArray as $habit) {
        if (strpos($habit, 'Lain-lain: ') === 0) {
            $badHabitsOther = substr($habit, strlen('Lain-lain: '));
            $badHabits[] = 'Lain-lain';
        } else {
            $badHabits[] = $habit;
        }
    }
}

// Process form submission for different sections
if (isPostRequest()) {
    if ($currentSection <= 2) {
        // Process Section 1 and 2 (Patient Data and Anamnesis)
        // Get form data
        $examinationDate = $_POST['examination_date'] ?? '';
        $medicalRecordNumber = $_POST['medical_record_number'] ?? '';
        $chiefComplaint = $_POST['chief_complaint'] ?? '';
        $medicalHistory = $_POST['medical_history'] ?? '';
        $badHabits = $_POST['bad_habits'] ?? [];
        $badHabitsOther = $_POST['bad_habits_other'] ?? '';
        
        // Validate required fields
        if (empty($examinationDate)) {
            $errorMessage = 'Tanggal pemeriksaan harus diisi.';
        } else if (empty($medicalRecordNumber)) {
            $errorMessage = 'Nomor rekam medis harus diisi.';
        } else if (empty($chiefComplaint)) {
            $errorMessage = 'Keluhan utama kasus ortodontik harus diisi.';
        } else if (empty($medicalHistory)) {
            $errorMessage = 'Riwayat kesehatan umum harus diisi.';
        } else {
            // Sanitize data
            $examinationDate = formatDateToMysql($examinationDate);
            $medicalRecordNumber = sanitize($medicalRecordNumber);
            $chiefComplaint = sanitize($chiefComplaint);
            $medicalHistory = sanitize($medicalHistory);
            
            // Process bad habits
            $badHabitsText = '';
            if (!empty($badHabits)) {
                $badHabitsText = implode(', ', $badHabits);
                
                // Add "other" if specified
                if (in_array('Lain-lain', $badHabits) && !empty($badHabitsOther)) {
                    $badHabitsText = str_replace('Lain-lain', 'Lain-lain: ' . sanitize($badHabitsOther), $badHabitsText);
                }
            }
            
            // Update medical record data
            $result = $db->query(
                "UPDATE medical_records SET 
                examination_date = ?s,
                medical_record_number = ?s,
                chief_complaint = ?s,
                medical_history = ?s,
                bad_habits = ?s,
                updated_by = ?i,
                updated_at = NOW()
                WHERE id = ?i",
                $examinationDate, $medicalRecordNumber, $chiefComplaint, $medicalHistory, 
                $badHabitsText, $_SESSION['user_id'], $recordId
            );
            
            if ($result) {
                $successMessage = 'Data rekam medis berhasil diperbarui.';
                
                // Refresh record data
                $record = $db->getRow("SELECT * FROM medical_records WHERE id = ?i", $recordId);
                
                // If the user clicked "Save and Next", move to the next section
                if (isset($_POST['save_and_next'])) {
                    redirect('edit.php?id=' . $recordId . '&section=' . ($currentSection + 1));
                }
            } else {
                $errorMessage = 'Terjadi kesalahan saat memperbarui data rekam medis.';
            }
        }
    } elseif ($currentSection == 3) {
        // Process Section 3 (Extra-Oral Examination)
        $faceShape = $_POST['face_shape'] ?? '';
        $verticalFaceBalance = $_POST['vertical_face_balance'] ?? '';
        $verticalFaceBalanceValue = ($verticalFaceBalance == 'Tidak Seimbang') ? ($_POST['vertical_face_balance_value'] ?? '') : null;
        $transversalFaceBalance = $_POST['transversal_face_balance'] ?? '';
        $transversalFaceBalanceValue = ($transversalFaceBalance == 'Tidak Seimbang') ? ($_POST['transversal_face_balance_value'] ?? '') : null;
        $chin = $_POST['chin'] ?? '';
        $upperDentalMidline = $_POST['upper_dental_midline'] ?? '';
        $upperDentalMidlineValue = ($upperDentalMidline == 'Ke Kiri' || $upperDentalMidline == 'Ke Kanan') ? ($_POST['upper_dental_midline_value'] ?? '') : null;
        $lowerDentalMidline = $_POST['lower_dental_midline'] ?? '';
        $lowerDentalMidlineValue = ($lowerDentalMidline == 'Ke Kiri' || $lowerDentalMidline == 'Ke Kanan') ? ($_POST['lower_dental_midline_value'] ?? '') : null;
        $upperLipTonus = $_POST['upper_lip_tonus'] ?? '';
        $lowerLipTonus = $_POST['lower_lip_tonus'] ?? '';
        $lipCompetence = $_POST['lip_competence'] ?? '';
        $elineProfile = $_POST['eline_profile'] ?? '';
        $upperLipToEline = $_POST['upper_lip_to_eline'] ?? '';
        $lowerLipToEline = $_POST['lower_lip_to_eline'] ?? '';
        
        // Sanitize inputs
        $faceShape = sanitize($faceShape);
        $verticalFaceBalance = sanitize($verticalFaceBalance);
        $transversalFaceBalance = sanitize($transversalFaceBalance);
        $chin = sanitize($chin);
        $upperDentalMidline = sanitize($upperDentalMidline);
        $lowerDentalMidline = sanitize($lowerDentalMidline);
        $upperLipTonus = sanitize($upperLipTonus);
        $lowerLipTonus = sanitize($lowerLipTonus);
        $lipCompetence = sanitize($lipCompetence);
        $elineProfile = sanitize($elineProfile);
        
        // If extra-oral examination record already exists, update it
        if ($extraOral) {
            $result = $db->query(
                "UPDATE extra_oral_examinations SET 
                face_shape = ?s, 
                vertical_face_balance = ?s, 
                vertical_face_balance_value = ?s, 
                transversal_face_balance = ?s, 
                transversal_face_balance_value = ?s, 
                chin = ?s, 
                upper_dental_midline = ?s, 
                upper_dental_midline_value = ?s, 
                lower_dental_midline = ?s, 
                lower_dental_midline_value = ?s, 
                upper_lip_tonus = ?s, 
                lower_lip_tonus = ?s, 
                lip_competence = ?s, 
                eline_profile = ?s, 
                upper_lip_to_eline = ?s, 
                lower_lip_to_eline = ?s, 
                updated_by = ?i, 
                updated_at = NOW() 
                WHERE id = ?i",
                $faceShape, $verticalFaceBalance, $verticalFaceBalanceValue, 
                $transversalFaceBalance, $transversalFaceBalanceValue, $chin, 
                $upperDentalMidline, $upperDentalMidlineValue, 
                $lowerDentalMidline, $lowerDentalMidlineValue, 
                $upperLipTonus, $lowerLipTonus, $lipCompetence, $elineProfile, 
                $upperLipToEline, $lowerLipToEline, 
                $_SESSION['user_id'], $extraOral['id']
            );
        } else {
            // Otherwise, create a new record
            $result = $db->query(
                "INSERT INTO extra_oral_examinations (
                    medical_record_id, face_shape, vertical_face_balance, vertical_face_balance_value, 
                    transversal_face_balance, transversal_face_balance_value, chin, 
                    upper_dental_midline, upper_dental_midline_value, 
                    lower_dental_midline, lower_dental_midline_value, 
                    upper_lip_tonus, lower_lip_tonus, lip_competence, eline_profile, 
                    upper_lip_to_eline, lower_lip_to_eline, 
                    created_by, created_at
                ) VALUES (
                    ?i, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?i, NOW()
                )",
                $recordId, $faceShape, $verticalFaceBalance, $verticalFaceBalanceValue, 
                $transversalFaceBalance, $transversalFaceBalanceValue, $chin, 
                $upperDentalMidline, $upperDentalMidlineValue, 
                $lowerDentalMidline, $lowerDentalMidlineValue, 
                $upperLipTonus, $lowerLipTonus, $lipCompetence, $elineProfile, 
                $upperLipToEline, $lowerLipToEline, 
                $_SESSION['user_id']
            );
        }
        
        if ($result) {
            // Update the medical record timestamp
            $db->query("UPDATE medical_records SET updated_at = NOW(), updated_by = ?i WHERE id = ?i", 
                $_SESSION['user_id'], $recordId);
            
            $successMessage = 'Data pemeriksaan ekstra oral berhasil disimpan.';
            
            // Refresh the extra-oral data
            $extraOral = $db->getRow("SELECT * FROM extra_oral_examinations WHERE medical_record_id = ?i", $recordId);
            
            // If the user clicked "Save and Next", move to the next section
            if (isset($_POST['save_and_next'])) {
                redirect('edit.php?id=' . $recordId . '&section=' . ($currentSection + 1));
            }
        } else {
            $errorMessage = 'Terjadi kesalahan saat menyimpan data pemeriksaan ekstra oral.';
        }
    } elseif ($currentSection == 4) {
        // Process Section 4 (Pemeriksaan Intra Oral)
        $oralHygiene = $_POST['oral_hygiene'] ?? '';
        $frenulumLabiiSuperior = $_POST['frenulum_labii_superior'] ?? '';
        $frenulumLabiiInferior = $_POST['frenulum_labii_inferior'] ?? '';
        $tongue = $_POST['tongue'] ?? '';
        $palatum = $_POST['palatum'] ?? '';
        $adenoid = $_POST['adenoid'] ?? '';
        $crowding = $_POST['crowding'] ?? '';
        $jawRelation = $_POST['jaw_relation'] ?? '';
        $diastema = $_POST['diastema'] ?? '';
        $crossbite = $_POST['crossbite'] ?? '';
        $toothShapeAbnormality = $_POST['tooth_shape_abnormality'] ?? '';
        $toothShapeAbnormalityRegion = $_POST['tooth_shape_abnormality_region'] ?? '';
        $toothNumberAbnormality = $_POST['tooth_number_abnormality'] ?? '';
        $toothNumberAbnormalityRegion = $_POST['tooth_number_abnormality_region'] ?? '';
        
        // Sanitize inputs
        $oralHygiene = sanitize($oralHygiene);
        $frenulumLabiiSuperior = sanitize($frenulumLabiiSuperior);
        $frenulumLabiiInferior = sanitize($frenulumLabiiInferior);
        $tongue = sanitize($tongue);
        $palatum = sanitize($palatum);
        $adenoid = sanitize($adenoid);
        $crowding = sanitize($crowding);
        $jawRelation = sanitize($jawRelation);
        $diastema = sanitize($diastema);
        $crossbite = sanitize($crossbite);
        $toothShapeAbnormalityRegion = sanitize($toothShapeAbnormalityRegion);
        $toothNumberAbnormalityRegion = sanitize($toothNumberAbnormalityRegion);
        
        // If intra-oral examination record already exists, update it
        if ($intraOral) {
            $result = $db->query(
                "UPDATE intra_oral_examinations SET 
                oral_hygiene = ?s,
                frenulum_labii_superior = ?s,
                frenulum_labii_inferior = ?s,
                tongue = ?s,
                palatum = ?s,
                adenoid = ?s,
                crowding = ?s,
                jaw_relation = ?s,
                diastema = ?s,
                crossbite = ?s,
                tooth_shape_abnormality = ?s,
                tooth_shape_abnormality_region = ?s,
                tooth_number_abnormality = ?s,
                tooth_number_abnormality_region = ?s,
                updated_by = ?i,
                updated_at = NOW()
                WHERE id = ?i",
                $oralHygiene, $frenulumLabiiSuperior, $frenulumLabiiInferior, 
                $tongue, $palatum, $adenoid, $crowding, $jawRelation, $diastema, $crossbite, 
                $toothShapeAbnormality, $toothShapeAbnormalityRegion, $toothNumberAbnormality, $toothNumberAbnormalityRegion, 
                $_SESSION['user_id'], $intraOral['id']
            );
        } else {
            // Otherwise, create a new record
            $result = $db->query(
                "INSERT INTO intra_oral_examinations (
                    medical_record_id, oral_hygiene, frenulum_labii_superior, frenulum_labii_inferior, 
                    tongue, palatum, adenoid, crowding, jaw_relation, diastema, crossbite, 
                    tooth_shape_abnormality, tooth_shape_abnormality_region, tooth_number_abnormality, tooth_number_abnormality_region, 
                    created_by, created_at
                ) VALUES (
                    ?i, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?i, NOW()
                )",
                $recordId, $oralHygiene, $frenulumLabiiSuperior, $frenulumLabiiInferior, 
                $tongue, $palatum, $adenoid, $crowding, $jawRelation, $diastema, $crossbite, 
                $toothShapeAbnormality, $toothShapeAbnormalityRegion, $toothNumberAbnormality, $toothNumberAbnormalityRegion, 
                $_SESSION['user_id']
            );
        }
        
        if ($result) {
            // Update the medical record timestamp
            $db->query("UPDATE medical_records SET updated_at = NOW(), updated_by = ?i WHERE id = ?i", 
                $_SESSION['user_id'], $recordId);
            
            $successMessage = 'Data pemeriksaan intra oral berhasil disimpan.';
            
            // Refresh the intra-oral data
            $intraOral = $db->getRow("SELECT * FROM intra_oral_examinations WHERE medical_record_id = ?i", $recordId);
            
            // If the user clicked "Save and Next", move to the next section
            if (isset($_POST['save_and_next'])) {
                redirect('edit.php?id=' . $recordId . '&section=' . ($currentSection + 1));
            }
        } else {
            $errorMessage = 'Terjadi kesalahan saat menyimpan data pemeriksaan intra oral.';
        }
    }
    
    // If the user clicked "Save and Complete", mark record as complete
    if (isset($_POST['save_and_complete'])) {
        $result = $db->query("UPDATE medical_records SET is_complete = 1, updated_at = NOW(), updated_by = ?i WHERE id = ?i", 
            $_SESSION['user_id'], $recordId);
        
        if ($result) {
            $successMessage = 'Rekam medis berhasil diselesaikan.';
            redirect('view.php?id=' . $recordId);
        } else {
            $errorMessage = 'Terjadi kesalahan saat menyelesaikan rekam medis.';
        }
    }
}

// Total number of sections
$totalSections = 13;

// Function to get section title
function getSectionTitle($section) {
    $titles = [
        1 => 'Data Diri Pasien',
        2 => 'Anamnesis',
        3 => 'Pemeriksaan Ekstra Oral',
        4 => 'Pemeriksaan Intra Oral',
        5 => 'Relasi Gigi Geligi',
        6 => 'Analisis Fungsional',
        7 => 'Analisis Radiologis',
        8 => 'Analisis Model',
        9 => 'Analisis Kebutuhan Ruang',
        10 => 'Diagnosis',
        11 => 'Rencana Terapi',
        12 => 'Gambar Alat',
        13 => 'Foto'
    ];
    
    return $titles[$section] ?? 'Bagian ' . $section;
}
?>
<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta content="width=device-width, initial-scale=1, maximum-scale=1, shrink-to-fit=no" name="viewport">
    <title>Edit Rekam Medis &mdash; Sistem Rekam Medis Ortodontik</title>

    <!-- General CSS Files -->
    <link rel="stylesheet" href="../assets/modules/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="../assets/modules/fontawesome/css/all.min.css">

    <!-- CSS Libraries -->
    <link rel="stylesheet" href="../assets/modules/bootstrap-daterangepicker/daterangepicker.css">
    <link rel="stylesheet" href="../assets/modules/select2/dist/css/select2.min.css">
    <link rel="stylesheet" href="../assets/modules/izitoast/css/iziToast.min.css">

    <!-- Template CSS -->
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/components.css">

    <style>
    .wizard-steps .wizard-step {
        flex: 1;
        text-align: center;
        padding: 10px;
        cursor: pointer;
    }

    .wizard-steps .wizard-step.active {
        font-weight: bold;
        background-color: #6777ef;
        color: white;
    }

    .wizard-steps .wizard-step.done {
        background-color: #63ed7a;
        color: white;
    }
    </style>
</head>

<body>
    <div id="app">
        <div class="main-wrapper main-wrapper-1">
            <!-- Header -->
            <?php include '../layouts/header.php'; ?>

            <!-- Sidebar -->
            <?php include '../layouts/sidebar.php'; ?>

            <!-- Main Content -->
            <div class="main-content">
                <section class="section">
                    <div class="section-header">
                        <h1>Edit Rekam Medis</h1>
                        <div class="section-header-breadcrumb">
                            <div class="breadcrumb-item active"><a href="../index.php">Dashboard</a></div>
                            <div class="breadcrumb-item"><a href="index.php">Rekam Medis</a></div>
                            <div class="breadcrumb-item">Edit Rekam Medis</div>
                        </div>
                    </div>

                    <div class="section-body">
                        <h2 class="section-title">Rekam Medis: <?php echo $patient['name']; ?></h2>
                        <p class="section-lead">
                            Lengkapi data rekam medis untuk pasien ini. Field dengan tanda
                            <?php echo getRequiredFieldMarker(); ?> wajib diisi.
                        </p>

                        <?php if (!empty($errorMessage)): ?>
                        <div class="alert alert-danger alert-dismissible show fade">
                            <div class="alert-body">
                                <button class="close" data-dismiss="alert">
                                    <span>×</span>
                                </button>
                                <?php echo $errorMessage; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <?php if (!empty($successMessage)): ?>
                        <div class="alert alert-success alert-dismissible show fade">
                            <div class="alert-body">
                                <button class="close" data-dismiss="alert">
                                    <span>×</span>
                                </button>
                                <?php echo $successMessage; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Navigation Steps -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <div class="wizard-steps">
                                    <?php for ($i = 1; $i <= $totalSections; $i++): ?>
                                    <div class="wizard-step <?php echo $i < $currentSection ? 'done' : ($i == $currentSection ? 'active' : ''); ?>"
                                        onclick="window.location='edit.php?id=<?php echo $recordId; ?>&section=<?php echo $i; ?>'">
                                        <div class="wizard-step-icon">
                                            <?php echo $i; ?>
                                        </div>
                                        <div class="wizard-step-label">
                                            <?php echo getSectionTitle($i); ?>
                                        </div>
                                    </div>
                                    <?php endfor; ?>
                                </div>
                            </div>
                        </div>

                        <form action="" method="POST">
                            <?php if ($currentSection == 1): ?>
                            <!-- First Section: Data Diri Pasien -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Data Diri Pasien</h4>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Tanggal Pemeriksaan
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control datepicker"
                                                            name="examination_date"
                                                            value="<?php echo formatDate($record['examination_date']); ?>"
                                                            required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Nomor Rekam Medis
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            name="medical_record_number"
                                                            value="<?php echo $record['medical_record_number']; ?>"
                                                            required readonly>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Tanggal Lahir
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo formatDate($patient['birth_date']); ?>"
                                                            readonly>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Usia <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo $patient['age']; ?> tahun" readonly>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Jenis Kelamin
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo $patient['gender']; ?>" readonly>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>

                            <?php if ($currentSection == 2): ?>
                            <!-- Second Section: Anamnesis -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Anamnesis</h4>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Keluhan Utama Kasus Ortodontik
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <textarea class="form-control" name="chief_complaint"
                                                            style="height: 100px;"
                                                            required><?php echo $record['chief_complaint']; ?></textarea>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Riwayat Kesehatan Umum yang berhubungan dengan maloklusi
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <textarea class="form-control" name="medical_history"
                                                            style="height: 100px;"
                                                            required><?php echo $record['medical_history']; ?></textarea>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Kebiasaan Buruk yang berhubungan dengan maloklusi</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]"
                                                                value="Tongue Thrust/Menjulurkan Lidah"
                                                                id="tongue_thrust"
                                                                <?php echo in_array("Tongue Thrust/Menjulurkan Lidah", $badHabits) ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="tongue_thrust">
                                                                Tongue Thrust/Menjulurkan Lidah
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]"
                                                                value="Mouth Breathing/Bernafas melalui mulut"
                                                                id="mouth_breathing"
                                                                <?php echo in_array("Mouth Breathing/Bernafas melalui mulut", $badHabits) ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="mouth_breathing">
                                                                Mouth Breathing/Bernafas melalui mulut
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Thumb Sucking/Menghisap Jari"
                                                                id="thumb_sucking"
                                                                <?php echo in_array("Thumb Sucking/Menghisap Jari", $badHabits) ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="thumb_sucking">
                                                                Thumb Sucking/Menghisap Jari
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Bruxism/Clenching"
                                                                id="bruxism"
                                                                <?php echo in_array("Bruxism/Clenching", $badHabits) ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="bruxism">
                                                                Bruxism/Clenching
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Lain-lain"
                                                                id="bad_habits_other_check"
                                                                <?php echo in_array("Lain-lain", $badHabits) ? 'checked' : ''; ?>
                                                                onclick="toggleOtherBadHabits()">
                                                            <label class="form-check-label"
                                                                for="bad_habits_other_check">
                                                                Dan lain-lain
                                                            </label>
                                                            <input type="text" class="form-control mt-2"
                                                                name="bad_habits_other" id="bad_habits_other_text"
                                                                style="display: <?php echo in_array("Lain-lain", $badHabits) ? 'block' : 'none'; ?>;"
                                                                value="<?php echo $badHabitsOther; ?>">
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>

                            <?php if ($currentSection == 3): ?>
                            <!-- Third Section: Pemeriksaan Ekstra Oral -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Pemeriksaan Ekstra Oral</h4>
                                        </div>
                                        <div class="card-body">
                                            <!-- Vertical Part -->
                                            <div class="section-title mt-0">Vertikal</div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Bentuk Wajah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="face_shape" id="mesofasial" value="Mesofasial"
                                                                <?php echo isset($extraOral['face_shape']) && $extraOral['face_shape'] == 'Mesofasial' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="mesofasial">
                                                                Mesofasial
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="face_shape" id="brakifasial" value="Brakifasial"
                                                                <?php echo isset($extraOral['face_shape']) && $extraOral['face_shape'] == 'Brakifasial' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="brakifasial">
                                                                Brakifasial
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="face_shape" id="dolikofasial" value="Dolikofasial"
                                                                <?php echo isset($extraOral['face_shape']) && $extraOral['face_shape'] == 'Dolikofasial' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="dolikofasial">
                                                                Dolikofasial
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Keseimbangan Wajah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="vertical_face_balance" id="balanced_vertical"
                                                                value="Seimbang"
                                                                <?php echo isset($extraOral['vertical_face_balance']) && $extraOral['vertical_face_balance'] == 'Seimbang' ? 'checked' : ''; ?>
                                                                onclick="toggleVerticalFaceBalanceInput(false)">
                                                            <label class="form-check-label" for="balanced_vertical">
                                                                Seimbang
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="vertical_face_balance" id="unbalanced_vertical"
                                                                value="Tidak Seimbang"
                                                                <?php echo isset($extraOral['vertical_face_balance']) && $extraOral['vertical_face_balance'] == 'Tidak Seimbang' ? 'checked' : ''; ?>
                                                                onclick="toggleVerticalFaceBalanceInput(true)">
                                                            <label class="form-check-label" for="unbalanced_vertical">
                                                                Tidak Seimbang
                                                            </label>
                                                        </div>
                                                        <div id="vertical_face_balance_input_container"
                                                            style="display: <?php echo isset($extraOral['vertical_face_balance']) && $extraOral['vertical_face_balance'] == 'Tidak Seimbang' ? 'block' : 'none'; ?>;">
                                                            <div class="input-group mt-2">
                                                                <input type="number" class="form-control"
                                                                    name="vertical_face_balance_value"
                                                                    id="vertical_face_balance_value" step="0.1" min="0"
                                                                    value="<?php echo isset($extraOral['vertical_face_balance_value']) ? $extraOral['vertical_face_balance_value'] : ''; ?>"
                                                                    placeholder="Masukkan nilai">
                                                                <div class="input-group-append">
                                                                    <span class="input-group-text">mm</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Transversal Part -->
                                            <div class="section-title mt-3">Transversal</div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Keseimbangan Wajah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="transversal_face_balance"
                                                                id="balanced_transversal" value="Seimbang"
                                                                <?php echo isset($extraOral['transversal_face_balance']) && $extraOral['transversal_face_balance'] == 'Seimbang' ? 'checked' : ''; ?>
                                                                onclick="toggleTransversalFaceBalanceInput(false)">
                                                            <label class="form-check-label" for="balanced_transversal">
                                                                Seimbang
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="transversal_face_balance"
                                                                id="unbalanced_transversal" value="Tidak Seimbang"
                                                                <?php echo isset($extraOral['transversal_face_balance']) && $extraOral['transversal_face_balance'] == 'Tidak Seimbang' ? 'checked' : ''; ?>
                                                                onclick="toggleTransversalFaceBalanceInput(true)">
                                                            <label class="form-check-label"
                                                                for="unbalanced_transversal">
                                                                Tidak Seimbang
                                                            </label>
                                                        </div>
                                                        <div id="transversal_face_balance_input_container"
                                                            style="display: <?php echo isset($extraOral['transversal_face_balance']) && $extraOral['transversal_face_balance'] == 'Tidak Seimbang' ? 'block' : 'none'; ?>;">
                                                            <div class="input-group mt-2">
                                                                <input type="number" class="form-control"
                                                                    name="transversal_face_balance_value"
                                                                    id="transversal_face_balance_value" step="0.1"
                                                                    min="0"
                                                                    value="<?php echo isset($extraOral['transversal_face_balance_value']) ? $extraOral['transversal_face_balance_value'] : ''; ?>"
                                                                    placeholder="Masukkan nilai">
                                                                <div class="input-group-append">
                                                                    <span class="input-group-text">mm</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Dagu</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="chin"
                                                                id="normal_chin" value="Normal"
                                                                <?php echo isset($extraOral['chin']) && $extraOral['chin'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_chin">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="chin"
                                                                id="right_chin" value="Miring ke Kanan"
                                                                <?php echo isset($extraOral['chin']) && $extraOral['chin'] == 'Miring ke Kanan' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="right_chin">
                                                                Miring ke Kanan
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="chin"
                                                                id="left_chin" value="Miring ke Kiri"
                                                                <?php echo isset($extraOral['chin']) && $extraOral['chin'] == 'Miring ke Kiri' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="left_chin">
                                                                Miring ke Kiri
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Garis Tengah Geligi Atas</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_dental_midline" id="normal_upper_midline"
                                                                value="Normal"
                                                                <?php echo isset($extraOral['upper_dental_midline']) && $extraOral['upper_dental_midline'] == 'Normal' ? 'checked' : ''; ?>
                                                                onclick="toggleUpperMidlineInput(false)">
                                                            <label class="form-check-label" for="normal_upper_midline">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_dental_midline" id="left_upper_midline"
                                                                value="Ke Kiri"
                                                                <?php echo isset($extraOral['upper_dental_midline']) && $extraOral['upper_dental_midline'] == 'Ke Kiri' ? 'checked' : ''; ?>
                                                                onclick="toggleUpperMidlineInput(true)">
                                                            <label class="form-check-label" for="left_upper_midline">
                                                                Ke Kiri
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_dental_midline" id="right_upper_midline"
                                                                value="Ke Kanan"
                                                                <?php echo isset($extraOral['upper_dental_midline']) && $extraOral['upper_dental_midline'] == 'Ke Kanan' ? 'checked' : ''; ?>
                                                                onclick="toggleUpperMidlineInput(true)">
                                                            <label class="form-check-label" for="right_upper_midline">
                                                                Ke Kanan
                                                            </label>
                                                        </div>
                                                        <div id="upper_midline_input_container"
                                                            style="display: <?php echo isset($extraOral['upper_dental_midline']) && ($extraOral['upper_dental_midline'] == 'Ke Kiri' || $extraOral['upper_dental_midline'] == 'Ke Kanan') ? 'block' : 'none'; ?>;">
                                                            <div class="input-group mt-2">
                                                                <input type="number" class="form-control"
                                                                    name="upper_dental_midline_value"
                                                                    id="upper_dental_midline_value" step="0.1" min="0"
                                                                    value="<?php echo isset($extraOral['upper_dental_midline_value']) ? $extraOral['upper_dental_midline_value'] : ''; ?>"
                                                                    placeholder="Masukkan nilai">
                                                                <div class="input-group-append">
                                                                    <span class="input-group-text">mm</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Garis Tengah Geligi Bawah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_dental_midline" id="normal_lower_midline"
                                                                value="Normal"
                                                                <?php echo isset($extraOral['lower_dental_midline']) && $extraOral['lower_dental_midline'] == 'Normal' ? 'checked' : ''; ?>
                                                                onclick="toggleLowerMidlineInput(false)">
                                                            <label class="form-check-label" for="normal_lower_midline">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_dental_midline" id="left_lower_midline"
                                                                value="Ke Kiri"
                                                                <?php echo isset($extraOral['lower_dental_midline']) && $extraOral['lower_dental_midline'] == 'Ke Kiri' ? 'checked' : ''; ?>
                                                                onclick="toggleLowerMidlineInput(true)">
                                                            <label class="form-check-label" for="left_lower_midline">
                                                                Ke Kiri
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_dental_midline" id="right_lower_midline"
                                                                value="Ke Kanan"
                                                                <?php echo isset($extraOral['lower_dental_midline']) && $extraOral['lower_dental_midline'] == 'Ke Kanan' ? 'checked' : ''; ?>
                                                                onclick="toggleLowerMidlineInput(true)">
                                                            <label class="form-check-label" for="right_lower_midline">
                                                                Ke Kanan
                                                            </label>
                                                        </div>
                                                        <div id="lower_midline_input_container"
                                                            style="display: <?php echo isset($extraOral['lower_dental_midline']) && ($extraOral['lower_dental_midline'] == 'Ke Kiri' || $extraOral['lower_dental_midline'] == 'Ke Kanan') ? 'block' : 'none'; ?>;">
                                                            <div class="input-group mt-2">
                                                                <input type="number" class="form-control"
                                                                    name="lower_dental_midline_value"
                                                                    id="lower_dental_midline_value" step="0.1" min="0"
                                                                    value="<?php echo isset($extraOral['lower_dental_midline_value']) ? $extraOral['lower_dental_midline_value'] : ''; ?>"
                                                                    placeholder="Masukkan nilai">
                                                                <div class="input-group-append">
                                                                    <span class="input-group-text">mm</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Lips Part -->
                                            <div class="section-title mt-3">Bibir</div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Tonus Bibir Atas</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_lip_tonus" id="normal_upper_lip"
                                                                value="Normal"
                                                                <?php echo isset($extraOral['upper_lip_tonus']) && $extraOral['upper_lip_tonus'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_upper_lip">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_lip_tonus" id="hypertonus_upper_lip"
                                                                value="Hipertonus"
                                                                <?php echo isset($extraOral['upper_lip_tonus']) && $extraOral['upper_lip_tonus'] == 'Hipertonus' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="hypertonus_upper_lip">
                                                                Hipertonus
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="upper_lip_tonus" id="hypotonus_upper_lip"
                                                                value="Hipotonus"
                                                                <?php echo isset($extraOral['upper_lip_tonus']) && $extraOral['upper_lip_tonus'] == 'Hipotonus' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="hypotonus_upper_lip">
                                                                Hipotonus
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Tonus Bibir Bawah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_lip_tonus" id="normal_lower_lip"
                                                                value="Normal"
                                                                <?php echo isset($extraOral['lower_lip_tonus']) && $extraOral['lower_lip_tonus'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_lower_lip">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_lip_tonus" id="hypertonus_lower_lip"
                                                                value="Hipertonus"
                                                                <?php echo isset($extraOral['lower_lip_tonus']) && $extraOral['lower_lip_tonus'] == 'Hipertonus' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="hypertonus_lower_lip">
                                                                Hipertonus
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lower_lip_tonus" id="hypotonus_lower_lip"
                                                                value="Hipotonus"
                                                                <?php echo isset($extraOral['lower_lip_tonus']) && $extraOral['lower_lip_tonus'] == 'Hipotonus' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="hypotonus_lower_lip">
                                                                Hipotonus
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Kompetensi Bibir</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lip_competence" id="competent_lip"
                                                                value="Kompeten"
                                                                <?php echo isset($extraOral['lip_competence']) && $extraOral['lip_competence'] == 'Kompeten' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="competent_lip">
                                                                Kompeten
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="lip_competence" id="incompetent_lip"
                                                                value="Tidak Kompeten"
                                                                <?php echo isset($extraOral['lip_competence']) && $extraOral['lip_competence'] == 'Tidak Kompeten' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="incompetent_lip">
                                                                Tidak Kompeten
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Profil E-Line</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="eline_profile" id="straight_eline" value="Lurus"
                                                                <?php echo isset($extraOral['eline_profile']) && $extraOral['eline_profile'] == 'Lurus' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="straight_eline">
                                                                Lurus
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="eline_profile" id="convex_eline" value="Cembung"
                                                                <?php echo isset($extraOral['eline_profile']) && $extraOral['eline_profile'] == 'Cembung' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="convex_eline">
                                                                Cembung
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="eline_profile" id="concave_eline" value="Cekung"
                                                                <?php echo isset($extraOral['eline_profile']) && $extraOral['eline_profile'] == 'Cekung' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="concave_eline">
                                                                Cekung
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Bibir Atas terhadap E-Line (mm)</label>
                                                        <div class="input-group">
                                                            <input type="number" class="form-control"
                                                                name="upper_lip_to_eline" step="0.1"
                                                                value="<?php echo isset($extraOral['upper_lip_to_eline']) ? $extraOral['upper_lip_to_eline'] : ''; ?>">
                                                            <div class="input-group-append">
                                                                <span class="input-group-text">mm</span>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Bibir Bawah terhadap E-Line (mm)</label>
                                                        <div class="input-group">
                                                            <input type="number" class="form-control"
                                                                name="lower_lip_to_eline" step="0.1"
                                                                value="<?php echo isset($extraOral['lower_lip_to_eline']) ? $extraOral['lower_lip_to_eline'] : ''; ?>">
                                                            <div class="input-group-append">
                                                                <span class="input-group-text">mm</span>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>

                            <?php if ($currentSection == 4): ?>
                            <!-- Fourth Section: Pemeriksaan Intra Oral -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Pemeriksaan Intra Oral</h4>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Kebersihan Mulut</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="oral_hygiene" id="good_hygiene" value="Baik"
                                                                <?php echo isset($intraOral['oral_hygiene']) && $intraOral['oral_hygiene'] == 'Baik' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="good_hygiene">
                                                                Baik
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="oral_hygiene" id="medium_hygiene" value="Sedang"
                                                                <?php echo isset($intraOral['oral_hygiene']) && $intraOral['oral_hygiene'] == 'Sedang' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="medium_hygiene">
                                                                Sedang
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="oral_hygiene" id="bad_hygiene" value="Buruk"
                                                                <?php echo isset($intraOral['oral_hygiene']) && $intraOral['oral_hygiene'] == 'Buruk' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="bad_hygiene">
                                                                Buruk
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Frenulum Labii Superior</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_superior" id="high_fls"
                                                                value="Tinggi"
                                                                <?php echo isset($intraOral['frenulum_labii_superior']) && $intraOral['frenulum_labii_superior'] == 'Tinggi' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="high_fls">
                                                                Tinggi
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_superior" id="medium_fls"
                                                                value="Sedang"
                                                                <?php echo isset($intraOral['frenulum_labii_superior']) && $intraOral['frenulum_labii_superior'] == 'Sedang' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="medium_fls">
                                                                Sedang
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_superior" id="low_fls"
                                                                value="Rendah"
                                                                <?php echo isset($intraOral['frenulum_labii_superior']) && $intraOral['frenulum_labii_superior'] == 'Rendah' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="low_fls">
                                                                Rendah
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Frenulum Labii Inferior</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_inferior" id="high_fli"
                                                                value="Tinggi"
                                                                <?php echo isset($intraOral['frenulum_labii_inferior']) && $intraOral['frenulum_labii_inferior'] == 'Tinggi' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="high_fli">
                                                                Tinggi
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_inferior" id="medium_fli"
                                                                value="Sedang"
                                                                <?php echo isset($intraOral['frenulum_labii_inferior']) && $intraOral['frenulum_labii_inferior'] == 'Sedang' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="medium_fli">
                                                                Sedang
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="frenulum_labii_inferior" id="low_fli"
                                                                value="Rendah"
                                                                <?php echo isset($intraOral['frenulum_labii_inferior']) && $intraOral['frenulum_labii_inferior'] == 'Rendah' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="low_fli">
                                                                Rendah
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Lidah</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="tongue"
                                                                id="normal_tongue" value="Normal"
                                                                <?php echo isset($intraOral['tongue']) && $intraOral['tongue'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_tongue">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="tongue"
                                                                id="macroglossia" value="Makroglosia"
                                                                <?php echo isset($intraOral['tongue']) && $intraOral['tongue'] == 'Makroglosia' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="macroglossia">
                                                                Makroglosia
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="tongue"
                                                                id="microglossia" value="Mikroglosia"
                                                                <?php echo isset($intraOral['tongue']) && $intraOral['tongue'] == 'Mikroglosia' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="microglossia">
                                                                Mikroglosia
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Palatum</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="palatum"
                                                                id="normal_palatum" value="Normal"
                                                                <?php echo isset($intraOral['palatum']) && $intraOral['palatum'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_palatum">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="palatum"
                                                                id="deep_palatum" value="Dalam"
                                                                <?php echo isset($intraOral['palatum']) && $intraOral['palatum'] == 'Dalam' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="deep_palatum">
                                                                Dalam
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="palatum"
                                                                id="abnormal_palatum" value="Kelainan"
                                                                <?php echo isset($intraOral['palatum']) && $intraOral['palatum'] == 'Kelainan' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="abnormal_palatum">
                                                                Kelainan
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Adenoid</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="adenoid"
                                                                id="normal_adenoid" value="Normal"
                                                                <?php echo isset($intraOral['adenoid']) && $intraOral['adenoid'] == 'Normal' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="normal_adenoid">
                                                                Normal
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="adenoid"
                                                                id="abnormal_adenoid" value="Kelainan"
                                                                <?php echo isset($intraOral['adenoid']) && $intraOral['adenoid'] == 'Kelainan' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="abnormal_adenoid">
                                                                Kelainan
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Crowding</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="crowding"
                                                                id="upper_crowding" value="Rahang Atas"
                                                                <?php echo isset($intraOral['crowding']) && $intraOral['crowding'] == 'Rahang Atas' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="upper_crowding">
                                                                Rahang Atas
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="crowding"
                                                                id="lower_crowding" value="Rahang Bawah"
                                                                <?php echo isset($intraOral['crowding']) && $intraOral['crowding'] == 'Rahang Bawah' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="lower_crowding">
                                                                Rahang Bawah
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="crowding"
                                                                id="both_crowding" value="Rahang Atas + Rahang Bawah"
                                                                <?php echo isset($intraOral['crowding']) && $intraOral['crowding'] == 'Rahang Atas + Rahang Bawah' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="both_crowding">
                                                                Rahang Atas + Rahang Bawah
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Hubungan Rahang</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="jaw_relation" id="class1_jaw" value="Kelas I"
                                                                <?php echo isset($intraOral['jaw_relation']) && $intraOral['jaw_relation'] == 'Kelas I' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="class1_jaw">
                                                                Kelas I
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="jaw_relation" id="class2_jaw" value="Kelas II"
                                                                <?php echo isset($intraOral['jaw_relation']) && $intraOral['jaw_relation'] == 'Kelas II' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="class2_jaw">
                                                                Kelas II
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio"
                                                                name="jaw_relation" id="class3_jaw" value="Kelas III"
                                                                <?php echo isset($intraOral['jaw_relation']) && $intraOral['jaw_relation'] == 'Kelas III' ? 'checked' : ''; ?>>
                                                            <label class="form-check-label" for="class3_jaw">
                                                                Kelas III
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Diastema</label>
                                                        <input type="text" class="form-control" name="diastema"
                                                            placeholder="Regio ..., ... mm"
                                                            value="<?php echo isset($intraOral['diastema']) ? $intraOral['diastema'] : ''; ?>">
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Crossbite</label>
                                                        <input type="text" class="form-control" name="crossbite"
                                                            placeholder="Regio ..."
                                                            value="<?php echo isset($intraOral['crossbite']) ? $intraOral['crossbite'] : ''; ?>">
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Kelainan Bentuk Gigi</label>
                                                        <div class="d-flex align-items-center mb-2">
                                                            <div class="form-check mr-3">
                                                                <input class="form-check-input" type="radio"
                                                                    name="tooth_shape_abnormality" id="yes_tooth_shape"
                                                                    value="Ada"
                                                                    <?php echo isset($intraOral['tooth_shape_abnormality']) && $intraOral['tooth_shape_abnormality'] == 'Ada' ? 'checked' : ''; ?>
                                                                    onclick="toggleToothShapeField(true)">
                                                                <label class="form-check-label" for="yes_tooth_shape">
                                                                    Ada
                                                                </label>
                                                            </div>
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio"
                                                                    name="tooth_shape_abnormality" id="no_tooth_shape"
                                                                    value="Tidak"
                                                                    <?php echo isset($intraOral['tooth_shape_abnormality']) && $intraOral['tooth_shape_abnormality'] == 'Tidak' ? 'checked' : ''; ?>
                                                                    onclick="toggleToothShapeField(false)">
                                                                <label class="form-check-label" for="no_tooth_shape">
                                                                    Tidak
                                                                </label>
                                                            </div>
                                                        </div>
                                                        <input type="text" class="form-control"
                                                            name="tooth_shape_abnormality_region"
                                                            id="tooth_shape_region" placeholder="Regio ..."
                                                            value="<?php echo isset($intraOral['tooth_shape_abnormality_region']) ? $intraOral['tooth_shape_abnormality_region'] : ''; ?>"
                                                            style="display: <?php echo isset($intraOral['tooth_shape_abnormality']) && $intraOral['tooth_shape_abnormality'] == 'Ada' ? 'block' : 'none'; ?>;">
                                                    </div>
                                                </div>

                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Kelainan Jumlah Gigi</label>
                                                        <div class="d-flex align-items-center mb-2">
                                                            <div class="form-check mr-3">
                                                                <input class="form-check-input" type="radio"
                                                                    name="tooth_number_abnormality"
                                                                    id="yes_tooth_number" value="Ada"
                                                                    <?php echo isset($intraOral['tooth_number_abnormality']) && $intraOral['tooth_number_abnormality'] == 'Ada' ? 'checked' : ''; ?>
                                                                    onclick="toggleToothNumberField(true)">
                                                                <label class="form-check-label" for="yes_tooth_number">
                                                                    Ada
                                                                </label>
                                                            </div>
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio"
                                                                    name="tooth_number_abnormality" id="no_tooth_number"
                                                                    value="Tidak"
                                                                    <?php echo isset($intraOral['tooth_number_abnormality']) && $intraOral['tooth_number_abnormality'] == 'Tidak' ? 'checked' : ''; ?>
                                                                    onclick="toggleToothNumberField(false)">
                                                                <label class="form-check-label" for="no_tooth_number">
                                                                    Tidak
                                                                </label>
                                                            </div>
                                                        </div>
                                                        <input type="text" class="form-control"
                                                            name="tooth_number_abnormality_region"
                                                            id="tooth_number_region" placeholder="Regio ..."
                                                            value="<?php echo isset($intraOral['tooth_number_abnormality_region']) ? $intraOral['tooth_number_abnormality_region'] : ''; ?>"
                                                            style="display: <?php echo isset($intraOral['tooth_number_abnormality']) && $intraOral['tooth_number_abnormality'] == 'Ada' ? 'block' : 'none'; ?>;">
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>

                            <div class="form-group row mb-4">
                                <div class="col-md-12 text-right">
                                    <?php if ($currentSection > 1): ?>
                                    <a href="edit.php?id=<?php echo $recordId; ?>&section=<?php echo $currentSection - 1; ?>"
                                        class="btn btn-light mr-2">
                                        <i class="fas fa-arrow-left"></i> Sebelumnya
                                    </a>
                                    <?php endif; ?>

                                    <button type="submit" name="save" class="btn btn-primary mr-2">
                                        <i class="fas fa-save"></i> Simpan
                                    </button>

                                    <?php if ($currentSection < $totalSections): ?>
                                    <button type="submit" name="save_and_next" class="btn btn-info">
                                        Simpan & Lanjutkan <i class="fas fa-arrow-right"></i>
                                    </button>
                                    <?php else: ?>
                                    <button type="submit" name="save_and_complete" class="btn btn-success">
                                        Simpan & Selesai <i class="fas fa-check"></i>
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </form>
                    </div>
                </section>
            </div>

            <!-- Footer -->
            <?php include '../layouts/footer.php'; ?>
        </div>
    </div>

    <!-- General JS Scripts -->
    <script src="../assets/modules/jquery.min.js"></script>
    <script src="../assets/modules/popper.js"></script>
    <script src="../assets/modules/tooltip.js"></script>
    <script src="../assets/modules/bootstrap/js/bootstrap.min.js"></script>
    <script src="../assets/modules/nicescroll/jquery.nicescroll.min.js"></script>
    <script src="../assets/modules/moment.min.js"></script>
    <script src="../assets/js/stisla.js"></script>

    <!-- JS Libraries -->
    <script src="../assets/modules/select2/dist/js/select2.full.min.js"></script>
    <script src="../assets/modules/bootstrap-daterangepicker/daterangepicker.js"></script>
    <script src="../assets/modules/izitoast/js/iziToast.min.js"></script>

    <!-- Page Specific JS File -->
    <script>
    $(document).ready(function() {
        // Initialize select2
        $('.select2').select2();

        // Initialize datepicker
        $('.datepicker').daterangepicker({
            locale: {
                format: 'DD-MM-YYYY',
                applyLabel: 'Pilih',
                cancelLabel: 'Batal',
                daysOfWeek: ['Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab'],
                monthNames: ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus',
                    'September', 'Oktober', 'November', 'Desember'
                ],
                firstDay: 1
            },
            singleDatePicker: true,
            showDropdowns: true,
            autoApply: true
        });

        <?php if (!empty($successMessage)): ?>
        // Show success notification
        iziToast.success({
            title: 'Sukses',
            message: '<?php echo $successMessage; ?>',
            position: 'topRight'
        });
        <?php endif; ?>
    });

    // Function to toggle "other" bad habits text field
    function toggleOtherBadHabits() {
        const checkbox = document.getElementById('bad_habits_other_check');
        const textField = document.getElementById('bad_habits_other_text');

        if (checkbox.checked) {
            textField.style.display = 'block';
            textField.focus();
        } else {
            textField.style.display = 'none';
            textField.value = '';
        }
    }
    </script>

    <!-- Template JS File -->
    <script src="../assets/js/scripts.js"></script>
    <script src="../assets/js/custom.js"></script>

    <!-- Additional JavaScript for Extra-Oral Examination -->
    <script>
    // Functions for Extra-Oral Examination toggle inputs
    function toggleVerticalFaceBalanceInput(show) {
        document.getElementById('vertical_face_balance_input_container').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('vertical_face_balance_value').value = '';
        }
    }

    function toggleTransversalFaceBalanceInput(show) {
        document.getElementById('transversal_face_balance_input_container').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('transversal_face_balance_value').value = '';
        }
    }

    function toggleUpperMidlineInput(show) {
        document.getElementById('upper_midline_input_container').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('upper_dental_midline_value').value = '';
        }
    }

    function toggleLowerMidlineInput(show) {
        document.getElementById('lower_midline_input_container').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('lower_dental_midline_value').value = '';
        }
    }

    // Functions for Intra-Oral Examination toggle inputs
    function toggleToothShapeField(show) {
        document.getElementById('tooth_shape_region').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('tooth_shape_region').value = '';
        }
    }

    function toggleToothNumberField(show) {
        document.getElementById('tooth_number_region').style.display = show ? 'block' : 'none';
        if (!show) {
            document.getElementById('tooth_number_region').value = '';
        }
    }
    </script>
</body>

</html>