<?php
// Include initialization file
require_once '../init.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}

// Check if user is a doctor (only doctors can create medical records)
if (!isDoctor()) {
    setFlashMessage('Anda tidak memiliki izin untuk membuat rekam medis.', 'danger');
    redirect('../index.php');
}

// Check if patient ID is provided
if (!isset($_GET['patient_id']) || empty($_GET['patient_id'])) {
    setFlashMessage('ID pasien tidak valid.', 'danger');
    redirect('../patients/index.php');
}

$patientId = intval($_GET['patient_id']);

// Get patient data
$patient = $db->getRow("SELECT 
    *,
    TIMESTAMPDIFF(YEAR, birth_date, CURDATE()) as age
    FROM patients 
    WHERE id = ?i", $patientId);

if (!$patient) {
    setFlashMessage('Pasien tidak ditemukan.', 'danger');
    redirect('../patients/index.php');
}

// Initialize variables
$errorMessage = '';
$successMessage = '';
$examinationDate = date('Y-m-d');
$doctorId = $_SESSION['user_id'];

// Process form submission (first section only)
if (isPostRequest()) {
    // Get form data
    $examinationDate = $_POST['examination_date'] ?? '';
    $medicalRecordNumber = $_POST['medical_record_number'] ?? '';
    $chiefComplaint = $_POST['chief_complaint'] ?? '';
    $medicalHistory = $_POST['medical_history'] ?? '';
    $badHabits = $_POST['bad_habits'] ?? [];
    $badHabitsOther = $_POST['bad_habits_other'] ?? '';
    
    // Validate required fields
    if (empty($examinationDate)) {
        $errorMessage = 'Tanggal pemeriksaan harus diisi.';
    } else if (empty($medicalRecordNumber)) {
        $errorMessage = 'Nomor rekam medis harus diisi.';
    } else if (empty($chiefComplaint)) {
        $errorMessage = 'Keluhan utama kasus ortodontik harus diisi.';
    } else if (empty($medicalHistory)) {
        $errorMessage = 'Riwayat kesehatan umum harus diisi.';
    } else {
        // Sanitize data
        $examinationDate = formatDateToMysql($examinationDate);
        $medicalRecordNumber = sanitize($medicalRecordNumber);
        $chiefComplaint = sanitize($chiefComplaint);
        $medicalHistory = sanitize($medicalHistory);
        
        // Process bad habits
        $badHabitsText = '';
        if (!empty($badHabits)) {
            $badHabitsText = implode(', ', $badHabits);
            
            // Add "other" if specified
            if (in_array('Lain-lain', $badHabits) && !empty($badHabitsOther)) {
                $badHabitsText = str_replace('Lain-lain', 'Lain-lain: ' . sanitize($badHabitsOther), $badHabitsText);
            }
        }
        
        // Insert basic medical record data
        $result = $db->query(
            "INSERT INTO medical_records (
                patient_id, 
                doctor_id, 
                examination_date, 
                medical_record_number,
                chief_complaint,
                medical_history,
                bad_habits,
                created_by,
                created_at
            ) VALUES (
                ?i, ?i, ?s, ?s, ?s, ?s, ?s, ?i, NOW()
            )",
            $patientId, $doctorId, $examinationDate, $medicalRecordNumber, 
            $chiefComplaint, $medicalHistory, $badHabitsText, $_SESSION['user_id']
        );
        
        if ($result) {
            $medicalRecordId = $db->insertId();
            
            // Log activity
            logActivity('Membuat rekam medis baru untuk pasien: ' . $patient['name']);
            
            // Redirect to edit page to continue with other sections
            setFlashMessage('Data dasar rekam medis berhasil disimpan. Silakan lengkapi bagian lainnya.', 'success');
            redirect('edit.php?id=' . $medicalRecordId);
        } else {
            $errorMessage = 'Terjadi kesalahan saat menyimpan data rekam medis.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta content="width=device-width, initial-scale=1, maximum-scale=1, shrink-to-fit=no" name="viewport">
    <title>Buat Rekam Medis &mdash; Sistem Rekam Medis Ortodontik</title>

    <!-- General CSS Files -->
    <link rel="stylesheet" href="../assets/modules/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="../assets/modules/fontawesome/css/all.min.css">

    <!-- CSS Libraries -->
    <link rel="stylesheet" href="../assets/modules/bootstrap-daterangepicker/daterangepicker.css">
    <link rel="stylesheet" href="../assets/modules/select2/dist/css/select2.min.css">
    <link rel="stylesheet" href="../assets/modules/izitoast/css/iziToast.min.css">

    <!-- Template CSS -->
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/components.css">
</head>

<body>
    <div id="app">
        <div class="main-wrapper main-wrapper-1">
            <!-- Header -->
            <?php include '../layouts/header.php'; ?>

            <!-- Sidebar -->
            <?php include '../layouts/sidebar.php'; ?>

            <!-- Main Content -->
            <div class="main-content">
                <section class="section">
                    <div class="section-header">
                        <h1>Buat Rekam Medis Baru</h1>
                        <div class="section-header-breadcrumb">
                            <div class="breadcrumb-item active"><a href="../index.php">Dashboard</a></div>
                            <div class="breadcrumb-item"><a href="../patients/index.php">Data Pasien</a></div>
                            <div class="breadcrumb-item"><a
                                    href="../patients/view.php?id=<?php echo $patientId; ?>">Detail Pasien</a></div>
                            <div class="breadcrumb-item">Buat Rekam Medis</div>
                        </div>
                    </div>

                    <div class="section-body">
                        <h2 class="section-title">Rekam Medis: <?php echo $patient['name']; ?></h2>
                        <p class="section-lead">
                            Isi data rekam medis untuk pasien. Field dengan tanda
                            <?php echo getRequiredFieldMarker(); ?> wajib diisi.
                        </p>

                        <?php if (!empty($errorMessage)): ?>
                        <div class="alert alert-danger alert-dismissible show fade">
                            <div class="alert-body">
                                <button class="close" data-dismiss="alert">
                                    <span>×</span>
                                </button>
                                <?php echo $errorMessage; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <?php if (!empty($successMessage)): ?>
                        <div class="alert alert-success alert-dismissible show fade">
                            <div class="alert-body">
                                <button class="close" data-dismiss="alert">
                                    <span>×</span>
                                </button>
                                <?php echo $successMessage; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <form action="" method="POST">
                            <!-- First Section: Data Diri Pasien -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Data Diri Pasien</h4>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Tanggal Pemeriksaan
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control datepicker"
                                                            name="examination_date"
                                                            value="<?php echo formatDate($examinationDate); ?>"
                                                            required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Nomor Rekam Medis
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            name="medical_record_number"
                                                            value="<?php echo $patient['medical_record_number']; ?>"
                                                            required readonly>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Tanggal Lahir
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo formatDate($patient['birth_date']); ?>"
                                                            readonly>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Usia <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo $patient['age']; ?> tahun" readonly>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label>Jenis Kelamin
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <input type="text" class="form-control"
                                                            value="<?php echo $patient['gender']; ?>" readonly>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Second Section: Anamnesis -->
                            <div class="row">
                                <div class="col-12">
                                    <div class="card">
                                        <div class="card-header">
                                            <h4>Anamnesis</h4>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Keluhan Utama Kasus Ortodontik
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <textarea class="form-control" name="chief_complaint"
                                                            style="height: 100px;" required></textarea>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Riwayat Kesehatan Umum yang berhubungan dengan maloklusi
                                                            <?php echo getRequiredFieldMarker(); ?></label>
                                                        <textarea class="form-control" name="medical_history"
                                                            style="height: 100px;" required></textarea>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="row">
                                                <div class="col-md-12">
                                                    <div class="form-group">
                                                        <label>Kebiasaan Buruk yang berhubungan dengan maloklusi</label>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]"
                                                                value="Tongue Thrust/Menjulurkan Lidah"
                                                                id="tongue_thrust">
                                                            <label class="form-check-label" for="tongue_thrust">
                                                                Tongue Thrust/Menjulurkan Lidah
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]"
                                                                value="Mouth Breathing/Bernafas melalui mulut"
                                                                id="mouth_breathing">
                                                            <label class="form-check-label" for="mouth_breathing">
                                                                Mouth Breathing/Bernafas melalui mulut
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Thumb Sucking/Menghisap Jari"
                                                                id="thumb_sucking">
                                                            <label class="form-check-label" for="thumb_sucking">
                                                                Thumb Sucking/Menghisap Jari
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Bruxism/Clenching"
                                                                id="bruxism">
                                                            <label class="form-check-label" for="bruxism">
                                                                Bruxism/Clenching
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox"
                                                                name="bad_habits[]" value="Lain-lain"
                                                                id="bad_habits_other_check"
                                                                onclick="toggleOtherBadHabits()">
                                                            <label class="form-check-label"
                                                                for="bad_habits_other_check">
                                                                Dan lain-lain
                                                            </label>
                                                            <input type="text" class="form-control mt-2"
                                                                name="bad_habits_other" id="bad_habits_other_text"
                                                                style="display: none;">
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="form-group row mb-4">
                                <div class="col-md-12 text-right">
                                    <a href="../patients/view.php?id=<?php echo $patientId; ?>"
                                        class="btn btn-secondary mr-2">Kembali</a>
                                    <button type="submit" class="btn btn-primary">Simpan & Lanjutkan</button>
                                </div>
                            </div>
                        </form>
                    </div>
                </section>
            </div>

            <!-- Footer -->
            <?php include '../layouts/footer.php'; ?>
        </div>
    </div>

    <!-- General JS Scripts -->
    <script src="../assets/modules/jquery.min.js"></script>
    <script src="../assets/modules/popper.js"></script>
    <script src="../assets/modules/tooltip.js"></script>
    <script src="../assets/modules/bootstrap/js/bootstrap.min.js"></script>
    <script src="../assets/modules/nicescroll/jquery.nicescroll.min.js"></script>
    <script src="../assets/modules/moment.min.js"></script>
    <script src="../assets/js/stisla.js"></script>

    <!-- JS Libraies -->
    <script src="../assets/modules/select2/dist/js/select2.full.min.js"></script>
    <script src="../assets/modules/bootstrap-daterangepicker/daterangepicker.js"></script>
    <script src="../assets/modules/izitoast/js/iziToast.min.js"></script>

    <!-- Page Specific JS File -->
    <script>
    $(document).ready(function() {
        // Initialize select2
        $('.select2').select2();

        // Initialize datepicker
        $('.datepicker').daterangepicker({
            locale: {
                format: 'DD-MM-YYYY',
                applyLabel: 'Pilih',
                cancelLabel: 'Batal',
                daysOfWeek: ['Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab'],
                monthNames: ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus',
                    'September', 'Oktober', 'November', 'Desember'
                ],
                firstDay: 1
            },
            singleDatePicker: true,
            showDropdowns: true,
            autoApply: true
        });
    });

    // Function to toggle "other" bad habits text field
    function toggleOtherBadHabits() {
        const checkbox = document.getElementById('bad_habits_other_check');
        const textField = document.getElementById('bad_habits_other_text');

        if (checkbox.checked) {
            textField.style.display = 'block';
            textField.focus();
        } else {
            textField.style.display = 'none';
            textField.value = '';
        }
    }
    </script>

    <!-- Template JS File -->
    <script src="../assets/js/scripts.js"></script>
    <script src="../assets/js/custom.js"></script>
</body>

</html>