<?php
/**
 * Handler for Section 7: Radiological Analysis (Cephalometric, Panoramic, and Other)
 */

// Initialize error flag
$hasError = false;

// Check if form was submitted
if (isPostRequest()) {
    // Create upload directory if it doesn't exist
    $uploadDir = '../uploads/radiological/';
    $recordUploadDir = $uploadDir . $recordId . '/';
    
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }
    
    if (!file_exists($recordUploadDir)) {
        mkdir($recordUploadDir, 0777, true);
    }
    
    // Check if this is a landmark-only save
    if (isset($_POST['save_landmarks_only']) && $_POST['save_landmarks_only'] == '1') {
        // Get or create radiological examination record
        $radiologicalExam = $db->getRow("SELECT * FROM radiological_examinations WHERE medical_record_id = ?i", $recordId);
        
        if (!$radiologicalExam) {
            // Create new radiological examination record
            $db->query("INSERT INTO radiological_examinations SET 
                medical_record_id = ?i, 
                created_at = NOW(), 
                created_by = ?i",
                $recordId, $_SESSION['user_id']);
        }
        
        // Get existing cephalometric analysis
        $existingCephalometricAnalysis = $db->getRow("SELECT id FROM cephalometric_analyses WHERE medical_record_id = ?i", $recordId);
        
        // Handle landmark data
        $landmarkData = $_POST['landmarks_data'] ?? '';
        if (!empty($landmarkData)) {
            // Try to decode to check if it's valid JSON
            $decodedData = json_decode($landmarkData);
            if (json_last_error() === JSON_ERROR_NONE) {
                // It's valid JSON, prepare data for update
                $cephData = [
                    'medical_record_id' => $recordId,
                    'landmark_data' => $landmarkData,
                    'updated_at' => date('Y-m-d H:i:s'),
                    'updated_by' => $_SESSION['user_id']
                ];
                
                if ($existingCephalometricAnalysis) {
                    // Update existing record
                    $result = $db->query("UPDATE cephalometric_analyses SET ?u WHERE id = ?i", 
                        $cephData, $existingCephalometricAnalysis['id']);
                } else {
                    // Create new record
                    $cephData['created_at'] = date('Y-m-d H:i:s');
                    $cephData['created_by'] = $_SESSION['user_id'];
                    
                    $result = $db->query("INSERT INTO cephalometric_analyses SET ?u", $cephData);
                }
                
                if ($result) {
                    $successMessage = 'Landmark berhasil disimpan.';
                } else {
                    $errorMessage = 'Gagal menyimpan landmark.';
                }
            } else {
                $errorMessage = 'Data landmark tidak valid.';
            }
        }
        
        // Don't redirect, just show the message
        return;
    }
    
    // Get or create radiological examination record
    $radiologicalExam = $db->getRow("SELECT * FROM radiological_examinations WHERE medical_record_id = ?i", $recordId);
    
    if (!$radiologicalExam) {
        // Create new radiological examination record
        $db->query("INSERT INTO radiological_examinations SET 
            medical_record_id = ?i, 
            created_at = NOW(), 
            created_by = ?i",
            $recordId, $_SESSION['user_id']);
    }
    
    // Update the radiological examination record
    $db->query("UPDATE radiological_examinations SET 
        updated_at = NOW(), 
        updated_by = ?i 
        WHERE medical_record_id = ?i",
        $_SESSION['user_id'], $recordId);
    
    // Handle Cephalometric Analysis
    // Process uploaded cephalometric image
    if (isset($_FILES['cephalometric_image']) && $_FILES['cephalometric_image']['error'] === UPLOAD_ERR_OK) {
        $cephFile = $_FILES['cephalometric_image'];
        $cephFileType = strtolower(pathinfo($cephFile['name'], PATHINFO_EXTENSION));
        
        // Check if image file is valid
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif'];
        
        if (in_array($cephFileType, $allowedExtensions)) {
            // Generate unique filename
            $cephFilename = 'cephalometric_' . uniqid() . '.' . $cephFileType;
            $cephTargetPath = $recordUploadDir . $cephFilename;
            
            // Move uploaded file
            if (move_uploaded_file($cephFile['tmp_name'], $cephTargetPath)) {
                // Successfully uploaded
                $cephalometricImagePath = $cephTargetPath;
            } else {
                $errorMessage = 'Gagal mengunggah gambar sefalometri.';
                $hasError = true;
            }
        } else {
            $errorMessage = 'Format file sefalometri tidak valid. Gunakan JPG, PNG, atau GIF.';
            $hasError = true;
        }
    }
    
    // Collect cephalometric analysis data
    $cephData = [
        'medical_record_id' => $recordId,
        'updated_at' => date('Y-m-d H:i:s'),
        'updated_by' => $_SESSION['user_id']
    ];
    
    // Handle landmark data - ensure it's valid JSON or set to NULL
    $landmarkData = $_POST['landmarks_data'] ?? '';
    if (!empty($landmarkData)) {
        // Try to decode to check if it's valid JSON
        $decodedData = json_decode($landmarkData);
        if (json_last_error() === JSON_ERROR_NONE) {
            // It's valid JSON, store as is
            $cephData['landmark_data'] = $landmarkData;
        } else {
            // Not valid JSON, store as JSON string
            $cephData['landmark_data'] = json_encode(['data' => $landmarkData]);
        }
    } else {
        // Empty data, set to NULL
        $cephData['landmark_data'] = NULL;
    }
    
    // Add image path if a new image was uploaded
    if (isset($cephalometricImagePath)) {
        $cephData['image_path'] = $cephalometricImagePath;
    }
    
    // Add cephalometric measurements
    $cephParams = [
        'sna_value' => $_POST['sna_value'] ?? null,
        'sna_conclusion' => $_POST['sna_conclusion'] ?? null,
        'snb_value' => $_POST['snb_value'] ?? null,
        'snb_conclusion' => $_POST['snb_conclusion'] ?? null,
        'anb_value' => $_POST['anb_value'] ?? null,
        'anb_conclusion' => $_POST['anb_conclusion'] ?? null,
        'napg_value' => $_POST['napg_value'] ?? null,
        'napg_conclusion' => $_POST['napg_conclusion'] ?? null,
        'mmpa_value' => $_POST['mmpa_value'] ?? null,
        'mmpa_conclusion' => $_POST['mmpa_conclusion'] ?? null,
        'fmpa_value' => $_POST['fmpa_value'] ?? null,
        'fmpa_conclusion' => $_POST['fmpa_conclusion'] ?? null,
        'fmia_value' => $_POST['fmia_value'] ?? null,
        'fmia_conclusion' => $_POST['fmia_conclusion'] ?? null,
        'impa_value' => $_POST['impa_value'] ?? null,
        'impa_conclusion' => $_POST['impa_conclusion'] ?? null,
        'i_sn_value' => $_POST['i_sn_value'] ?? null,
        'i_sn_conclusion' => $_POST['i_sn_conclusion'] ?? null,
        'i_mxpi_value' => $_POST['i_mxpi_value'] ?? null,
        'i_mxpi_conclusion' => $_POST['i_mxpi_conclusion'] ?? null,
        'ui_na_value' => $_POST['ui_na_value'] ?? null,
        'ui_na_conclusion' => $_POST['ui_na_conclusion'] ?? null,
        'li_nb_value' => $_POST['li_nb_value'] ?? null,
        'li_nb_conclusion' => $_POST['li_nb_conclusion'] ?? null,
        'interincisal_angle' => $_POST['interincisal_angle_value'] ?? null,
        'interincisal_angle_conclusion' => $_POST['interincisal_angle_conclusion'] ?? null,
        'e_line_upper_lip' => $_POST['eline_upper_lip_value'] ?? null,
        'eline_upper_lip_conclusion' => $_POST['eline_upper_lip_conclusion'] ?? null,
        'e_line_lower_lip' => $_POST['eline_lower_lip_value'] ?? null,
        'eline_lower_lip_conclusion' => $_POST['eline_lower_lip_conclusion'] ?? null,
        'cephalometric_conclusion' => $_POST['cephalometric_conclusion'] ?? null
    ];
    
    // Sanitize and add to cephData
    foreach ($cephParams as $key => $value) {
        if (!is_null($value)) {
            $cephData[$key] = sanitize($value);
        }
    }
    
    // Save cephalometric analysis data
    $existingCephalometricAnalysis = $db->getRow("SELECT id FROM cephalometric_analyses WHERE medical_record_id = ?i", $recordId);
    
    if ($existingCephalometricAnalysis) {
        // Update existing record
        $result = $db->query("UPDATE cephalometric_analyses SET ?u WHERE id = ?i", 
            $cephData, $existingCephalometricAnalysis['id']);
    } else {
        // Create new record
        $cephData['created_at'] = date('Y-m-d H:i:s');
        $cephData['created_by'] = $_SESSION['user_id'];
        
        $result = $db->query("INSERT INTO cephalometric_analyses SET ?u", $cephData);
    }
    
    // Handle Panoramic Analysis
    // Process uploaded panoramic image
    if (isset($_FILES['panoramic_image']) && $_FILES['panoramic_image']['error'] === UPLOAD_ERR_OK) {
        $panoFile = $_FILES['panoramic_image'];
        $panoFileType = strtolower(pathinfo($panoFile['name'], PATHINFO_EXTENSION));
        
        // Check if image file is valid
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif'];
        
        if (in_array($panoFileType, $allowedExtensions)) {
            // Generate unique filename
            $panoFilename = 'panoramic_' . uniqid() . '.' . $panoFileType;
            $panoTargetPath = $recordUploadDir . $panoFilename;
            
            // Move uploaded file
            if (move_uploaded_file($panoFile['tmp_name'], $panoTargetPath)) {
                // Successfully uploaded
                $panoramicImagePath = $panoTargetPath;
            } else {
                $errorMessage = 'Gagal mengunggah gambar panoramik.';
                $hasError = true;
            }
        } else {
            $errorMessage = 'Format file panoramik tidak valid. Gunakan JPG, PNG, atau GIF.';
            $hasError = true;
        }
    }
    
    // Collect panoramic analysis data
    $panoData = [
        'medical_record_id' => $recordId,
        'bone_condition' => sanitize($_POST['bone_condition'] ?? ''),
        'periodontal_tissue' => sanitize($_POST['periodontal_condition'] ?? ''),
        'caries' => sanitize($_POST['caries'] ?? ''),
        'missing_teeth' => sanitize($_POST['missing_teeth'] ?? ''),
        'agenesis' => sanitize($_POST['agenesis'] ?? ''),
        'agenesis_region' => sanitize($_POST['agenesis_regio'] ?? ''),
        'impaction' => sanitize($_POST['impaction'] ?? ''),
        'supernumerary_teeth' => sanitize($_POST['supernumerary_teeth'] ?? ''),
        'other_findings' => sanitize($_POST['panoramic_other'] ?? ''),
        'eruption_sequence_top_right' => sanitize($_POST['eruption_upper_right'] ?? ''),
        'eruption_sequence_top_left' => sanitize($_POST['eruption_upper_left'] ?? ''),
        'eruption_sequence_bottom_right' => sanitize($_POST['eruption_lower_right'] ?? ''),
        'eruption_sequence_bottom_left' => sanitize($_POST['eruption_lower_left'] ?? ''),
        'updated_at' => date('Y-m-d H:i:s'),
        'updated_by' => $_SESSION['user_id']
    ];
    
    // Add image path if a new image was uploaded
    if (isset($panoramicImagePath)) {
        $panoData['image_path'] = $panoramicImagePath;
    }
    
    // Save panoramic analysis data
    $existingPanoramicAnalysis = $db->getRow("SELECT id FROM panoramic_analyses WHERE medical_record_id = ?i", $recordId);
    
    if ($existingPanoramicAnalysis) {
        // Update existing record
        $result = $db->query("UPDATE panoramic_analyses SET ?u WHERE id = ?i", 
            $panoData, $existingPanoramicAnalysis['id']);
    } else {
        // Create new record
        $panoData['created_at'] = date('Y-m-d H:i:s');
        $panoData['created_by'] = $_SESSION['user_id'];
        
        $result = $db->query("INSERT INTO panoramic_analyses SET ?u", $panoData);
    }
    
    // Handle Other Radiographs
    // Process uploaded other radiograph files
    if (isset($_FILES['other_radiograph']) && !empty($_FILES['other_radiograph']['name'][0])) {
        $otherFiles = $_FILES['other_radiograph'];
        $fileCount = count($otherFiles['name']);
        
        for ($i = 0; $i < $fileCount; $i++) {
            if ($otherFiles['error'][$i] === UPLOAD_ERR_OK) {
                $otherFileType = strtolower(pathinfo($otherFiles['name'][$i], PATHINFO_EXTENSION));
                
                // Check if file is valid
                $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'dcm'];
                
                if (in_array($otherFileType, $allowedExtensions)) {
                    // Generate unique filename
                    $otherFilename = 'other_' . uniqid() . '.' . $otherFileType;
                    $otherTargetPath = $recordUploadDir . $otherFilename;
                    
                    // Move uploaded file
                    if (move_uploaded_file($otherFiles['tmp_name'][$i], $otherTargetPath)) {
                        // Successfully uploaded, save to database
                        $otherData = [
                            'medical_record_id' => $recordId,
                            'image_path' => $otherTargetPath,
                            'image_type' => getRadiographType($otherFileType),
                            'notes' => '',
                            'created_at' => date('Y-m-d H:i:s'),
                            'created_by' => $_SESSION['user_id']
                        ];
                        
                        $db->query("INSERT INTO other_radiographs SET ?u", $otherData);
                    } else {
                        $errorMessage = 'Gagal mengunggah salah satu file radiograf.';
                        $hasError = true;
                    }
                } else {
                    $errorMessage = 'Format file radiograf tidak valid. Gunakan JPG, PNG, GIF, PDF, atau DICOM.';
                    $hasError = true;
                }
            }
        }
    }
    
    // Handle deleted radiographs
    if (isset($_POST['delete_radiograph']) && is_array($_POST['delete_radiograph'])) {
        foreach ($_POST['delete_radiograph'] as $radiographId) {
            if (!empty($radiographId)) {
                // Get file path before deleting
                $radiograph = $db->getRow("SELECT image_path FROM other_radiographs WHERE id = ?i", $radiographId);
                
                if ($radiograph && file_exists($radiograph['image_path'])) {
                    // Delete file from filesystem
                    unlink($radiograph['image_path']);
                }
                
                // Delete record from database
                $db->query("DELETE FROM other_radiographs WHERE id = ?i", $radiographId);
            }
        }
    }
    
    // Update the parent medical record's updated_at and updated_by
    $db->query("UPDATE medical_records SET updated_at = NOW(), updated_by = ?i WHERE id = ?i", 
        $_SESSION['user_id'], $recordId);
    
    if (!$hasError) {
        // Success message
        $successMessage = 'Data Analisis Radiologis berhasil disimpan.';
        
        // Handle save and continue
        if (isset($_POST['save_and_next'])) {
            setFlashMessage('Data Analisis Radiologis berhasil disimpan.', 'success');
            redirect("edit.php?id={$recordId}&section=" . ($currentSection + 1));
        }
    }
}

/**
 * Helper function to determine the radiograph type based on the file extension
 */
function getRadiographType($extension) {
    switch($extension) {
        case 'dcm':
            return 'CBCT';
        case 'pdf':
            return 'Dental';
        default:
            // Default to dental for images
            return 'Dental';
    }
} 