<?php
/**
 * Handler for Section 9: Analisis Kebutuhan Ruang (Space Analysis)
 */

// Initialize error flag
$hasError = false;

// Check if form was submitted
if (isPostRequest()) {
    // Get selected analysis methods
    $analysisMethods = $_POST['analysis_methods'] ?? [];
    
    // Prepare space analysis data
    $spaceData = [
        'medical_record_id' => $recordId,
        'updated_at' => date('Y-m-d H:i:s'),
        'updated_by' => $_SESSION['user_id']
    ];
    
    // Handle space analysis conclusion
    if (isset($_POST['space_analysis_conclusion'])) {
        $spaceData['space_analysis_conclusion'] = sanitize($_POST['space_analysis_conclusion']);
    }
    
    // Process Moyers Analysis
    if (in_array('moyers', $analysisMethods)) {
        // Process and save basic Moyers data
        $moyers32_31_41_42_width = isset($_POST['moyers_32_31_41_42_width']) ? sanitize($_POST['moyers_32_31_41_42_width']) : null;
        $moyers_gender = isset($_POST['moyers_gender']) ? sanitize($_POST['moyers_gender']) : null;
        $moyers_max_left_available = isset($_POST['moyers_max_left_available']) ? sanitize($_POST['moyers_max_left_available']) : null;
        $moyers_max_right_available = isset($_POST['moyers_max_right_available']) ? sanitize($_POST['moyers_max_right_available']) : null;
        $moyers_mand_left_available = isset($_POST['moyers_mand_left_available']) ? sanitize($_POST['moyers_mand_left_available']) : null;
        $moyers_mand_right_available = isset($_POST['moyers_mand_right_available']) ? sanitize($_POST['moyers_mand_right_available']) : null;
        
        $spaceData['moyers_32_31_41_42_width'] = $moyers32_31_41_42_width;
        $spaceData['moyers_gender'] = $moyers_gender;
        $spaceData['moyers_max_left_available'] = $moyers_max_left_available;
        $spaceData['moyers_max_right_available'] = $moyers_max_right_available;
        $spaceData['moyers_mand_left_available'] = $moyers_mand_left_available;
        $spaceData['moyers_mand_right_available'] = $moyers_mand_right_available;
        
        // Calculate required space based on Moyers table (75% probability)
        if (!empty($moyers32_31_41_42_width) && !empty($moyers_gender)) {
            $moyersRequiredSpace = calculateMoyersRequiredSpace($moyers32_31_41_42_width, $moyers_gender);
            
            // Store calculated values
            $spaceData['moyers_max_left_required'] = $moyersRequiredSpace['maxillary'];
            $spaceData['moyers_max_right_required'] = $moyersRequiredSpace['maxillary'];
            $spaceData['moyers_mand_left_required'] = $moyersRequiredSpace['mandibular'];
            $spaceData['moyers_mand_right_required'] = $moyersRequiredSpace['mandibular'];
        }
    }
    
    // Process Kesling Analysis
    if (in_array('kesling', $analysisMethods)) {
        // Process and save Kesling data
        $kesling_max_left_required = isset($_POST['kesling_max_left_required']) ? sanitize($_POST['kesling_max_left_required']) : null;
        $kesling_max_right_required = isset($_POST['kesling_max_right_required']) ? sanitize($_POST['kesling_max_right_required']) : null;
        $kesling_mand_left_required = isset($_POST['kesling_mand_left_required']) ? sanitize($_POST['kesling_mand_left_required']) : null;
        $kesling_mand_right_required = isset($_POST['kesling_mand_right_required']) ? sanitize($_POST['kesling_mand_right_required']) : null;
        
        $kesling_max_left_available = isset($_POST['kesling_max_left_available']) ? sanitize($_POST['kesling_max_left_available']) : null;
        $kesling_max_right_available = isset($_POST['kesling_max_right_available']) ? sanitize($_POST['kesling_max_right_available']) : null;
        $kesling_mand_left_available = isset($_POST['kesling_mand_left_available']) ? sanitize($_POST['kesling_mand_left_available']) : null;
        $kesling_mand_right_available = isset($_POST['kesling_mand_right_available']) ? sanitize($_POST['kesling_mand_right_available']) : null;
        
        $spaceData['kesling_max_left_required'] = $kesling_max_left_required;
        $spaceData['kesling_max_right_required'] = $kesling_max_right_required;
        $spaceData['kesling_mand_left_required'] = $kesling_mand_left_required;
        $spaceData['kesling_mand_right_required'] = $kesling_mand_right_required;
        
        $spaceData['kesling_max_left_available'] = $kesling_max_left_available;
        $spaceData['kesling_max_right_available'] = $kesling_max_right_available;
        $spaceData['kesling_mand_left_available'] = $kesling_mand_left_available;
        $spaceData['kesling_mand_right_available'] = $kesling_mand_right_available;
    }
    
    // Process Lundstrum (ALD) Analysis
    if (in_array('lundstrum', $analysisMethods)) {
        // Process Maxilla segments
        $maxSegments = ['s1', 's2', 's3', 's4', 's5', 's6'];
        foreach ($maxSegments as $segment) {
            $required = isset($_POST["lundstrum_max_{$segment}_required"]) ? sanitize($_POST["lundstrum_max_{$segment}_required"]) : null;
            $available = isset($_POST["lundstrum_max_{$segment}_available"]) ? sanitize($_POST["lundstrum_max_{$segment}_available"]) : null;
            
            $spaceData["lundstrum_max_{$segment}_required"] = $required;
            $spaceData["lundstrum_max_{$segment}_available"] = $available;
        }
        
        // Process Mandibula segments
        $mandSegments = ['s1', 's2', 's3', 's4', 's5', 's6'];
        foreach ($mandSegments as $segment) {
            $required = isset($_POST["lundstrum_mand_{$segment}_required"]) ? sanitize($_POST["lundstrum_mand_{$segment}_required"]) : null;
            $available = isset($_POST["lundstrum_mand_{$segment}_available"]) ? sanitize($_POST["lundstrum_mand_{$segment}_available"]) : null;
            
            $spaceData["lundstrum_mand_{$segment}_required"] = $required;
            $spaceData["lundstrum_mand_{$segment}_available"] = $available;
        }
    }
    
    // Process Ponts Analysis
    if (in_array('ponts', $analysisMethods)) {
        // Save input values
        $ponts_mpv = isset($_POST['ponts_mpv']) ? sanitize($_POST['ponts_mpv']) : null;
        $ponts_mmv = isset($_POST['ponts_mmv']) ? sanitize($_POST['ponts_mmv']) : null;
        $ponts_si = isset($_POST['ponts_si']) ? sanitize($_POST['ponts_si']) : null;
        
        $spaceData['ponts_mpv'] = $ponts_mpv;
        $spaceData['ponts_mmv'] = $ponts_mmv;
        $spaceData['ponts_si'] = $ponts_si;
        
        // Calculate and save CPV and CMV values (for future reference if needed)
        if (!empty($ponts_si)) {
            $spaceData['ponts_cpv'] = ($ponts_si / 80) * 100;
            $spaceData['ponts_cmv'] = ($ponts_si / 64) * 100;
        }
    }
    
    // Process Howe Analysis
    if (in_array('howe', $analysisMethods)) {
        // Save input values
        $howe_pmbaw = isset($_POST['howe_pmbaw']) ? sanitize($_POST['howe_pmbaw']) : null;
        $howe_ttm = isset($_POST['howe_ttm']) ? sanitize($_POST['howe_ttm']) : null;
        
        $spaceData['howe_pmbaw'] = $howe_pmbaw;
        $spaceData['howe_ttm'] = $howe_ttm;
        
        // Calculate and save ratio value (for future reference if needed)
        if (!empty($howe_pmbaw) && !empty($howe_ttm) && $howe_ttm != 0) {
            $spaceData['howe_index'] = ($howe_pmbaw / $howe_ttm) * 100;
        }
    }
    
    // Process space methods for each quadrant
    $quadrants = ['max_right', 'max_left', 'mand_right', 'mand_left'];
    foreach ($quadrants as $quadrant) {
        if (isset($_POST[$quadrant . '_method']) && is_array($_POST[$quadrant . '_method'])) {
            $spaceData[$quadrant . '_space_method'] = json_encode($_POST[$quadrant . '_method']);
        } else {
            $spaceData[$quadrant . '_space_method'] = json_encode([]);
        }
        
        // Process extraction teeth if available
        if (isset($_POST[$quadrant . '_extract']) && is_array($_POST[$quadrant . '_extract'])) {
            $spaceData[$quadrant . '_extraction_teeth'] = json_encode($_POST[$quadrant . '_extract']);
        } else {
            $spaceData[$quadrant . '_extraction_teeth'] = json_encode([]);
        }
    }
    
    // Check if record already exists
    $existingSpaceAnalysis = $db->getRow("SELECT id FROM space_analyses WHERE medical_record_id = ?i", $recordId);
    
    if ($existingSpaceAnalysis) {
        // Update existing record
        $result = $db->query(
            "UPDATE space_analyses SET ?u WHERE id = ?i",
            $spaceData, $existingSpaceAnalysis['id']
        );
    } else {
        // Create new record
        $spaceData['created_at'] = date('Y-m-d H:i:s');
        $spaceData['created_by'] = $_SESSION['user_id'];
        
        $result = $db->query(
            "INSERT INTO space_analyses SET ?u",
            $spaceData
        );
    }
    
    // Handle result
    if ($result) {
        $successMessage = 'Analisis kebutuhan ruang berhasil disimpan.';
        
        // Handle navigation
        if (isset($_POST['save_and_next'])) {
            // Set flash message before redirecting
            setFlashMessage($successMessage, 'success');
            redirect('edit.php?id=' . $recordId . '&section=' . ($currentSection + 1));
        }
    } else {
        $errorMessage = 'Terjadi kesalahan saat menyimpan analisis kebutuhan ruang.';
        $hasError = true;
    }
}

/**
 * Calculate required space based on Moyers table (75% probability)
 * @param float $width Sum of mandibular incisor widths (32, 31, 41, 42)
 * @param string $gender Gender (Laki-Laki or Perempuan)
 * @return array Required space for maxillary and mandibular arches
 */
function calculateMoyersRequiredSpace($width, $gender) {
    // Moyers table for 75% probability (sum of mandibular incisors vs. required space)
    $moyersTable = [
        // Width range => [maxillary required, mandibular required]
        '19.5' => ['maxillary' => 20.6, 'mandibular' => 20.1],
        '20.0' => ['maxillary' => 20.9, 'mandibular' => 20.4],
        '20.5' => ['maxillary' => 21.2, 'mandibular' => 20.7],
        '21.0' => ['maxillary' => 21.5, 'mandibular' => 21.0],
        '21.5' => ['maxillary' => 21.8, 'mandibular' => 21.3],
        '22.0' => ['maxillary' => 22.0, 'mandibular' => 21.6],
        '22.5' => ['maxillary' => 22.3, 'mandibular' => 21.9],
        '23.0' => ['maxillary' => 22.6, 'mandibular' => 22.2],
        '23.5' => ['maxillary' => 22.9, 'mandibular' => 22.5],
        '24.0' => ['maxillary' => 23.1, 'mandibular' => 22.8],
        '24.5' => ['maxillary' => 23.4, 'mandibular' => 23.1],
        '25.0' => ['maxillary' => 23.7, 'mandibular' => 23.4],
        '25.5' => ['maxillary' => 24.0, 'mandibular' => 23.7],
        '26.0' => ['maxillary' => 24.2, 'mandibular' => 24.0],
        '26.5' => ['maxillary' => 24.5, 'mandibular' => 24.3],
        '27.0' => ['maxillary' => 24.8, 'mandibular' => 24.6],
    ];
    
    // Round the width to nearest 0.5
    $roundedWidth = round($width * 2) / 2;
    $roundedWidth = number_format($roundedWidth, 1);
    
    // Make sure width is in the range of the table
    if ($roundedWidth < 19.5) {
        $roundedWidth = '19.5';
    } elseif ($roundedWidth > 27.0) {
        $roundedWidth = '27.0';
    }
    
    // Get values from table (default to the specified width if exact match exists)
    if (isset($moyersTable[$roundedWidth])) {
        return $moyersTable[$roundedWidth];
    }
    
    // If we don't have an exact match, we should interpolate, but for simplicity,
    // we'll use the closest value in the table
    $keys = array_keys($moyersTable);
    $closest = $keys[0];
    $minDiff = abs(floatval($roundedWidth) - floatval($closest));
    
    foreach ($keys as $key) {
        $diff = abs(floatval($roundedWidth) - floatval($key));
        if ($diff < $minDiff) {
            $closest = $key;
            $minDiff = $diff;
        }
    }
    
    return $moyersTable[$closest];
}
?> 