<?php
// Section 12 - Gambar Alat
// Get existing appliance image data if it exists
$applianceImage = $db->getRow("SELECT * FROM appliance_images WHERE medical_record_id = ?i", $recordId);

// Initialize canvas data from database or default
$canvasData = $applianceImage['canvas_data'] ?? '';
?>

<div class="section-body">
    <h2 class="section-title">Gambar Alat</h2>

    <?php if (!empty($errorMessage)): ?>
    <div class="alert alert-danger"><?= $errorMessage ?></div>
    <?php endif; ?>

    <?php if (!empty($successMessage)): ?>
    <div class="alert alert-success"><?= $successMessage ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h4>Desain Alat Ortodonti</h4>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-9">
                    <div class="mb-3">
                        <button type="button" id="clear-canvas" class="btn btn-danger">
                            <i class="fas fa-trash"></i> Hapus Gambar
                        </button>
                        <button type="button" id="save-canvas" class="btn btn-primary">
                            <i class="fas fa-save"></i> Simpan Gambar
                        </button>
                        <button type="button" id="toggle-draw-mode" class="btn btn-info">
                            <i class="fas fa-pencil-alt"></i> Mode Menggambar
                        </button>
                        <input type="color" id="brush-color" value="#000000" class="ml-2">
                        <select id="brush-size" class="custom-select ml-2" style="width: 80px">
                            <option value="1">1px</option>
                            <option value="3" selected>3px</option>
                            <option value="5">5px</option>
                            <option value="8">8px</option>
                            <option value="12">12px</option>
                        </select>
                        <button type="button" id="toggle-eraser" class="btn btn-outline-secondary ml-2"
                            title="Penghapus">
                            <i class="fas fa-eraser"></i>
                        </button>
                    </div>

                    <div class="canvas-container"
                        style="position: relative; border: 1px solid #ccc; background-color: #f9f9f9;">
                        <canvas id="appliance-canvas" width="873" height="1205"
                            style="border: 1px solid #ccc; background-color: white;"></canvas>
                    </div>

                    <!-- Hidden field to store canvas data -->
                    <input type="hidden" id="canvas-data" name="canvas_data"
                        value="<?= htmlspecialchars($canvasData) ?>">
                </div>

                <div class="col-md-3">
                    <div class="card">
                        <div class="card-header">
                            <h4>Komponen Alat</h4>
                        </div>
                        <div class="card-body">
                            <div class="btn-group-vertical w-100">
                                <button type="button" class="btn btn-outline-primary mb-2 add-appliance"
                                    data-type="t-spring">
                                    <i class="fas fa-plus-circle"></i> T-Spring
                                </button>
                                <button type="button" class="btn btn-outline-primary mb-2 add-appliance"
                                    data-type="finger-spring">
                                    <i class="fas fa-plus-circle"></i> Finger Spring
                                </button>
                                <button type="button" class="btn btn-outline-primary mb-2 add-appliance"
                                    data-type="z-spring">
                                    <i class="fas fa-plus-circle"></i> Z-Spring
                                </button>
                                <button type="button" class="btn btn-outline-primary mb-2 add-appliance"
                                    data-type="adams">
                                    <i class="fas fa-plus-circle"></i> Adam's
                                </button>
                                <button type="button" class="btn btn-outline-primary mb-2 add-appliance"
                                    data-type="labial-bow">
                                    <i class="fas fa-plus-circle"></i> Labial Bow
                                </button>
                            </div>

                            <hr>

                            <div class="mt-3">
                                <h6>Alat Aktif</h6>
                                <ul id="active-appliances-list" class="list-group">
                                    <!-- Active appliances will be listed here -->
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.appliance-overlay {
    position: absolute;
    cursor: move;
    border: 2px dashed transparent;
}

.appliance-overlay.selected {
    border: 2px dashed #007bff;
}

.appliance-overlay .resize-handle {
    position: absolute;
    width: 10px;
    height: 10px;
    background-color: #007bff;
    border-radius: 50%;
    bottom: -5px;
    right: -5px;
    cursor: nwse-resize;
    display: none;
    /* Hide by default */
}

.appliance-overlay .remove-btn {
    position: absolute;
    top: -10px;
    right: -10px;
    width: 20px;
    height: 20px;
    background-color: #dc3545;
    color: white;
    border-radius: 50%;
    text-align: center;
    line-height: 20px;
    font-size: 12px;
    cursor: pointer;
    display: none;
    /* Hide by default */
}

/* Controls toolbar that appears when an overlay is selected */
.appliance-overlay .controls-toolbar {
    position: absolute;
    top: -40px;
    left: 50%;
    transform: translateX(-50%);
    background-color: rgba(255, 255, 255, 0.9);
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 3px;
    display: none;
    /* Hide by default */
    white-space: nowrap;
    box-shadow: 0 2px 5px rgba(0, 0, 0, 0.15);
}

.appliance-overlay .control-btn {
    width: 24px;
    height: 24px;
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 3px;
    margin: 0 2px;
    padding: 0;
    line-height: 1;
    text-align: center;
    cursor: pointer;
    font-size: 12px;
}

.appliance-overlay .control-btn:hover {
    background-color: #e9ecef;
}

/* Show controls only for selected overlays */
.appliance-overlay.selected .resize-handle,
.appliance-overlay.selected .remove-btn,
.appliance-overlay.selected .controls-toolbar {
    display: block;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const canvas = document.getElementById('appliance-canvas');
    const ctx = canvas.getContext('2d');
    const canvasContainer = document.querySelector('.canvas-container');
    const clearButton = document.getElementById('clear-canvas');
    const saveButton = document.getElementById('save-canvas');
    const toggleDrawModeButton = document.getElementById('toggle-draw-mode');
    const toggleEraserButton = document.getElementById('toggle-eraser');
    const brushColorInput = document.getElementById('brush-color');
    const brushSizeInput = document.getElementById('brush-size');
    const canvasDataInput = document.getElementById('canvas-data');
    const activeAppliancesList = document.getElementById('active-appliances-list');

    let isDrawing = false;
    let drawMode = false;
    let eraserMode = false;
    let lastX = 0;
    let lastY = 0;
    let brushColor = brushColorInput.value;
    let brushSize = parseInt(brushSizeInput.value);
    let appliances = [];
    let selectedApplianceIndex = -1;

    // Load appliance images
    const applianceImages = {
        'base-mouth': '/orthobaru/assets/img/appliances/mouth-base.png',
        't-spring': '/orthobaru/assets/img/appliances/t-spring.png',
        'finger-spring': '/orthobaru/assets/img/appliances/finger-spring.png',
        'z-spring': '/orthobaru/assets/img/appliances/z-spring.png',
        'adams': '/orthobaru/assets/img/appliances/adams.png',
        'labial-bow': '/orthobaru/assets/img/appliances/labial-bow.png'
    };

    // Initialize canvas with base mouth image
    const baseImage = new Image();
    baseImage.src = applianceImages['base-mouth'];
    baseImage.onload = function() {
        ctx.drawImage(baseImage, 0, 0, canvas.width, canvas.height);
        loadCanvasData();
    };

    // Load saved canvas data if exists
    function loadCanvasData() {
        if (canvasDataInput.value) {
            try {
                const data = JSON.parse(canvasDataInput.value);

                // Load drawing data
                if (data.imageData) {
                    const img = new Image();
                    img.onload = function() {
                        ctx.drawImage(img, 0, 0);
                    };
                    img.src = data.imageData;
                }

                // Load appliances
                if (data.appliances && Array.isArray(data.appliances)) {
                    appliances = data.appliances;
                    renderAppliances();
                    updateAppliancesList();
                }
            } catch (e) {
                console.error('Error loading canvas data:', e);
            }
        }
    }

    // Save canvas data
    function saveCanvasData() {
        const data = {
            imageData: canvas.toDataURL(),
            appliances: appliances
        };

        canvasDataInput.value = JSON.stringify(data);

        // Send data to server using AJAX
        const formData = new FormData();
        formData.append('action', 'save_canvas');
        formData.append('record_id', <?= $recordId ?>);
        formData.append('canvas_data', JSON.stringify(data));

        fetch('includes/section_handlers/section12_handler.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    iziToast.success({
                        title: 'Sukses',
                        message: 'Gambar alat berhasil disimpan',
                        position: 'topRight'
                    });
                } else {
                    iziToast.error({
                        title: 'Error',
                        message: 'Gagal menyimpan gambar alat',
                        position: 'topRight'
                    });
                }
            })
            .catch(error => {
                console.error('Error saving canvas data:', error);
                iziToast.error({
                    title: 'Error',
                    message: 'Terjadi kesalahan saat menyimpan gambar alat',
                    position: 'topRight'
                });
            });
    }

    // Add appliance to canvas
    function addAppliance(type) {
        const img = new Image();
        img.src = applianceImages[type];
        img.onload = function() {
            // Default position in center of canvas
            const applianceObj = {
                type: type,
                x: (canvas.width - 100) / 2,
                y: (canvas.height - 100) / 2,
                width: 100,
                height: 100 * (img.height / img.width),
                rotation: 0,
                scaleX: 1,
                scaleY: 1
            };

            appliances.push(applianceObj);
            createApplianceOverlay(applianceObj, appliances.length - 1);
            updateAppliancesList();
        };
    }

    // Create overlay for appliance
    function createApplianceOverlay(appliance, index) {
        const overlay = document.createElement('div');
        overlay.className = 'appliance-overlay';
        overlay.dataset.index = index;
        overlay.style.width = appliance.width + 'px';
        overlay.style.height = appliance.height + 'px';
        overlay.style.left = appliance.x + 'px';
        overlay.style.top = appliance.y + 'px';

        // Apply rotation and scaling
        const transform = [];
        if (appliance.rotation) {
            transform.push(`rotate(${appliance.rotation}deg)`);
        }
        if (appliance.scaleX !== 1) {
            transform.push(`scaleX(${appliance.scaleX})`);
        }
        if (appliance.scaleY !== 1) {
            transform.push(`scaleY(${appliance.scaleY})`);
        }
        if (transform.length > 0) {
            overlay.style.transform = transform.join(' ');
        }

        // Create resize handle
        const resizeHandle = document.createElement('div');
        resizeHandle.className = 'resize-handle';
        overlay.appendChild(resizeHandle);

        // Create remove button
        const removeBtn = document.createElement('div');
        removeBtn.className = 'remove-btn';
        removeBtn.innerHTML = 'x';
        removeBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            removeAppliance(index);
        });
        overlay.appendChild(removeBtn);

        // Create controls toolbar
        const controlsToolbar = document.createElement('div');
        controlsToolbar.className = 'controls-toolbar';

        // Stretch horizontally button
        const stretchHBtn = document.createElement('button');
        stretchHBtn.className = 'control-btn';
        stretchHBtn.innerHTML = '↔️';
        stretchHBtn.title = 'Stretch Horizontally';
        stretchHBtn.type = 'button'; // Prevent form submission
        stretchHBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            const newWidth = appliance.width * 1.2;
            overlay.style.width = newWidth + 'px';
            appliances[index].width = newWidth;
        });
        controlsToolbar.appendChild(stretchHBtn);

        // Stretch vertically button
        const stretchVBtn = document.createElement('button');
        stretchVBtn.className = 'control-btn';
        stretchVBtn.innerHTML = '↕️';
        stretchVBtn.title = 'Stretch Vertically';
        stretchVBtn.type = 'button'; // Prevent form submission
        stretchVBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            const newHeight = appliance.height * 1.2;
            overlay.style.height = newHeight + 'px';
            appliances[index].height = newHeight;
        });
        controlsToolbar.appendChild(stretchVBtn);

        // Rotate left button
        const rotateLeftBtn = document.createElement('button');
        rotateLeftBtn.className = 'control-btn';
        rotateLeftBtn.innerHTML = '↺';
        rotateLeftBtn.title = 'Rotate Left';
        rotateLeftBtn.type = 'button'; // Prevent form submission
        rotateLeftBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            const newRotation = (appliance.rotation || 0) - 15;
            appliances[index].rotation = newRotation;
            updateTransform(overlay, appliances[index]);
        });
        controlsToolbar.appendChild(rotateLeftBtn);

        // Rotate right button
        const rotateRightBtn = document.createElement('button');
        rotateRightBtn.className = 'control-btn';
        rotateRightBtn.innerHTML = '↻';
        rotateRightBtn.title = 'Rotate Right';
        rotateRightBtn.type = 'button'; // Prevent form submission
        rotateRightBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            const newRotation = (appliance.rotation || 0) + 15;
            appliances[index].rotation = newRotation;
            updateTransform(overlay, appliances[index]);
        });
        controlsToolbar.appendChild(rotateRightBtn);

        // Mirror horizontally button
        const mirrorHBtn = document.createElement('button');
        mirrorHBtn.className = 'control-btn';
        mirrorHBtn.innerHTML = '⇆';
        mirrorHBtn.title = 'Mirror Horizontally';
        mirrorHBtn.type = 'button'; // Prevent form submission
        mirrorHBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            appliances[index].scaleX = appliances[index].scaleX * -1;
            updateTransform(overlay, appliances[index]);
        });
        controlsToolbar.appendChild(mirrorHBtn);

        // Mirror vertically button
        const mirrorVBtn = document.createElement('button');
        mirrorVBtn.className = 'control-btn';
        mirrorVBtn.innerHTML = '⇅';
        mirrorVBtn.title = 'Mirror Vertically';
        mirrorVBtn.type = 'button'; // Prevent form submission
        mirrorVBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            appliances[index].scaleY = appliances[index].scaleY * -1;
            updateTransform(overlay, appliances[index]);
        });
        controlsToolbar.appendChild(mirrorVBtn);

        // Reset transformations button
        const resetBtn = document.createElement('button');
        resetBtn.className = 'control-btn';
        resetBtn.innerHTML = '⟲';
        resetBtn.title = 'Reset Transformations';
        resetBtn.type = 'button'; // Prevent form submission
        resetBtn.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent default action
            e.stopPropagation();
            appliances[index].rotation = 0;
            appliances[index].scaleX = 1;
            appliances[index].scaleY = 1;
            updateTransform(overlay, appliances[index]);
        });
        controlsToolbar.appendChild(resetBtn);

        overlay.appendChild(controlsToolbar);

        // Add image
        const img = document.createElement('img');
        img.src = applianceImages[appliance.type];
        img.style.width = '100%';
        img.style.height = '100%';
        img.draggable = false;
        overlay.appendChild(img);

        // Make overlay draggable
        overlay.addEventListener('mousedown', function(e) {
            if (e.target !== removeBtn && !e.target.classList.contains('control-btn')) {
                e.stopPropagation(); // Prevent click from propagating to canvas or document
                selectAppliance(index);
                const startX = e.clientX;
                const startY = e.clientY;
                const startLeft = parseInt(overlay.style.left);
                const startTop = parseInt(overlay.style.top);

                const mouseMoveHandler = function(e) {
                    const newLeft = startLeft + e.clientX - startX;
                    const newTop = startTop + e.clientY - startY;
                    overlay.style.left = newLeft + 'px';
                    overlay.style.top = newTop + 'px';
                    appliances[index].x = newLeft;
                    appliances[index].y = newTop;
                };

                const mouseUpHandler = function() {
                    document.removeEventListener('mousemove', mouseMoveHandler);
                    document.removeEventListener('mouseup', mouseUpHandler);
                };

                document.addEventListener('mousemove', mouseMoveHandler);
                document.addEventListener('mouseup', mouseUpHandler);
            }
        });

        // Make resize handle work
        resizeHandle.addEventListener('mousedown', function(e) {
            e.stopPropagation();
            const startX = e.clientX;
            const startY = e.clientY;
            const startWidth = parseInt(overlay.style.width);
            const startHeight = parseInt(overlay.style.height);

            const mouseMoveHandler = function(e) {
                const newWidth = startWidth + e.clientX - startX;
                // Maintain aspect ratio
                const aspectRatio = appliances[index].height / appliances[index].width;
                const newHeight = newWidth * aspectRatio;

                overlay.style.width = newWidth + 'px';
                overlay.style.height = newHeight + 'px';
                appliances[index].width = newWidth;
                appliances[index].height = newHeight;
            };

            const mouseUpHandler = function() {
                document.removeEventListener('mousemove', mouseMoveHandler);
                document.removeEventListener('mouseup', mouseUpHandler);
            };

            document.addEventListener('mousemove', mouseMoveHandler);
            document.addEventListener('mouseup', mouseUpHandler);
        });

        canvasContainer.appendChild(overlay);
    }

    // Helper function to update transform style
    function updateTransform(overlay, appliance) {
        const transform = [];
        if (appliance.rotation) {
            transform.push(`rotate(${appliance.rotation}deg)`);
        }
        if (appliance.scaleX !== 1) {
            transform.push(`scaleX(${appliance.scaleX})`);
        }
        if (appliance.scaleY !== 1) {
            transform.push(`scaleY(${appliance.scaleY})`);
        }
        overlay.style.transform = transform.length > 0 ? transform.join(' ') : 'none';
    }

    // Select an appliance
    function selectAppliance(index) {
        // Deselect all
        document.querySelectorAll('.appliance-overlay').forEach(overlay => {
            overlay.classList.remove('selected');
        });

        // Select the one clicked
        selectedApplianceIndex = index;
        document.querySelector(`.appliance-overlay[data-index="${index}"]`).classList.add('selected');
    }

    // Deselect all appliances
    function deselectAllAppliances() {
        document.querySelectorAll('.appliance-overlay').forEach(overlay => {
            overlay.classList.remove('selected');
        });
        selectedApplianceIndex = -1;
    }

    // Remove an appliance
    function removeAppliance(index) {
        appliances.splice(index, 1);
        renderAppliances();
        updateAppliancesList();
    }

    // Re-render all appliance overlays
    function renderAppliances() {
        // Remove all existing overlays
        document.querySelectorAll('.appliance-overlay').forEach(overlay => {
            overlay.remove();
        });

        // Recreate all overlays
        appliances.forEach((appliance, index) => {
            createApplianceOverlay(appliance, index);
        });
    }

    // Update the active appliances list
    function updateAppliancesList() {
        activeAppliancesList.innerHTML = '';

        if (appliances.length === 0) {
            const emptyItem = document.createElement('li');
            emptyItem.className = 'list-group-item text-muted';
            emptyItem.textContent = 'Tidak ada alat yang aktif';
            activeAppliancesList.appendChild(emptyItem);
            return;
        }

        appliances.forEach((appliance, index) => {
            const item = document.createElement('li');
            item.className = 'list-group-item d-flex justify-content-between align-items-center';

            // Convert type to display name
            let displayName = '';
            switch (appliance.type) {
                case 't-spring':
                    displayName = 'T-Spring';
                    break;
                case 'finger-spring':
                    displayName = 'Finger Spring';
                    break;
                case 'z-spring':
                    displayName = 'Z-Spring';
                    break;
                case 'adams':
                    displayName = 'Adam\'s';
                    break;
                case 'labial-bow':
                    displayName = 'Labial Bow';
                    break;
            }

            item.textContent = displayName;

            const removeBtn = document.createElement('button');
            removeBtn.className = 'btn btn-sm btn-danger';
            removeBtn.innerHTML = '<i class="fas fa-trash"></i>';
            removeBtn.addEventListener('click', function() {
                removeAppliance(index);
            });

            item.appendChild(removeBtn);
            activeAppliancesList.appendChild(item);
        });
    }

    // Drawing functionality
    function startDrawing(e) {
        if (!drawMode) return;

        isDrawing = true;
        [lastX, lastY] = [
            e.offsetX || e.touches[0].clientX - canvas.offsetLeft,
            e.offsetY || e.touches[0].clientY - canvas.offsetTop
        ];
    }

    function draw(e) {
        if (!isDrawing || !drawMode) return;

        const x = e.offsetX || e.touches[0].clientX - canvas.offsetLeft;
        const y = e.offsetY || e.touches[0].clientY - canvas.offsetTop;

        ctx.lineJoin = 'round';
        ctx.lineCap = 'round';
        ctx.lineWidth = brushSize;

        // Set the composite operation based on whether we're in eraser mode
        if (eraserMode) {
            ctx.globalCompositeOperation = 'destination-out';
            ctx.strokeStyle = 'rgba(0,0,0,1)'; // Doesn't matter for eraser
        } else {
            ctx.globalCompositeOperation = 'source-over';
            ctx.strokeStyle = brushColor;
        }

        ctx.beginPath();
        ctx.moveTo(lastX, lastY);
        ctx.lineTo(x, y);
        ctx.stroke();

        [lastX, lastY] = [x, y];
    }

    function stopDrawing() {
        isDrawing = false;
        // Reset composite operation when done drawing
        ctx.globalCompositeOperation = 'source-over';
    }

    // Set up event listeners
    canvas.addEventListener('mousedown', function(e) {
        // When clicking on the canvas (not on an overlay), deselect all appliances
        deselectAllAppliances();
        startDrawing(e);
    });
    canvas.addEventListener('mousemove', draw);
    canvas.addEventListener('mouseup', stopDrawing);
    canvas.addEventListener('mouseout', stopDrawing);

    // Add a click event handler to the document to deselect appliances when clicking outside
    document.addEventListener('click', function(e) {
        // Check if the click was on an appliance overlay or its children
        const clickedOnOverlay = e.target.closest('.appliance-overlay');
        if (!clickedOnOverlay) {
            deselectAllAppliances();
        }
    });

    // Touch events for mobile
    canvas.addEventListener('touchstart', startDrawing);
    canvas.addEventListener('touchmove', draw);
    canvas.addEventListener('touchend', stopDrawing);

    // Add appliance buttons
    document.querySelectorAll('.add-appliance').forEach(button => {
        button.addEventListener('click', function() {
            const type = this.dataset.type;
            addAppliance(type);
        });
    });

    // Clear canvas button
    clearButton.addEventListener('click', function() {
        if (confirm('Apakah Anda yakin ingin menghapus semua gambar?')) {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.drawImage(baseImage, 0, 0, canvas.width, canvas.height);

            appliances = [];
            renderAppliances();
            updateAppliancesList();
        }
    });

    // Save canvas button
    saveButton.addEventListener('click', saveCanvasData);

    // Toggle draw mode
    toggleDrawModeButton.addEventListener('click', function() {
        drawMode = !drawMode;
        this.classList.toggle('btn-info', !drawMode);
        this.classList.toggle('btn-success', drawMode);

        if (drawMode) {
            this.innerHTML = '<i class="fas fa-pencil-alt"></i> Mode Menggambar (Aktif)';
            canvas.style.cursor = 'crosshair';

            // If turning on draw mode, ensure eraser button reflects correct state
            toggleEraserButton.classList.toggle('btn-secondary', eraserMode);
            toggleEraserButton.classList.toggle('btn-outline-secondary', !eraserMode);
        } else {
            this.innerHTML = '<i class="fas fa-pencil-alt"></i> Mode Menggambar';
            canvas.style.cursor = 'default';

            // Turn off eraser mode when exiting draw mode
            eraserMode = false;
            toggleEraserButton.classList.remove('btn-secondary');
            toggleEraserButton.classList.add('btn-outline-secondary');
        }
    });

    // Toggle eraser mode
    toggleEraserButton.addEventListener('click', function() {
        // Only toggle if in draw mode
        if (drawMode) {
            eraserMode = !eraserMode;
            this.classList.toggle('btn-secondary', eraserMode);
            this.classList.toggle('btn-outline-secondary', !eraserMode);

            if (eraserMode) {
                canvas.style.cursor =
                    'url(\'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20"><circle cx="10" cy="10" r="5" fill="white" stroke="black" stroke-width="1"/></svg>\') 10 10, auto';
            } else {
                canvas.style.cursor = 'crosshair';
            }
        }
    });

    // Brush color change
    brushColorInput.addEventListener('change', function() {
        brushColor = this.value;
        // Turn off eraser mode when changing color
        if (eraserMode) {
            eraserMode = false;
            toggleEraserButton.classList.remove('btn-secondary');
            toggleEraserButton.classList.add('btn-outline-secondary');
            canvas.style.cursor = 'crosshair';
        }
    });

    // Brush size change
    brushSizeInput.addEventListener('change', function() {
        brushSize = parseInt(this.value);
    });

    // Initialize the appliance list
    updateAppliancesList();

    // Add event listener to the form to save canvas data before submission
    // Need to find the form that contains our canvas element
    setTimeout(function() {
        // Using setTimeout to ensure all DOM elements are fully loaded
        const formElement = document.querySelector('form[method="POST"]');
        if (formElement) {
            console.log('Form found, attaching submit handler');
            formElement.addEventListener('submit', function(e) {
                console.log('Form submit detected, saving canvas data');
                // Update canvas data in the hidden field before form submission
                const data = {
                    imageData: canvas.toDataURL(),
                    appliances: appliances
                };

                canvasDataInput.value = JSON.stringify(data);
                // Form submission continues normally
            });
        } else {
            console.error('Form element not found');
        }
    }, 500); // Half-second delay to ensure DOM is fully loaded
});
</script>