<?php
// Section 7: Radiological Analysis (Cephalometric, Panoramic, and Other)

// Get the radiological examination data for this record
$radiologicalExam = $db->getRow("SELECT * FROM radiological_examinations WHERE medical_record_id = ?i", $recordId);
$radiologicalExamId = isset($radiologicalExam['id']) ? $radiologicalExam['id'] : null;

// Get the cephalometric analysis data for this record
$cephalometricAnalysis = $db->getRow("SELECT * FROM cephalometric_analyses WHERE medical_record_id = ?i", $recordId);

// Get the panoramic analysis data for this record
$panoramicAnalysis = $db->getRow("SELECT * FROM panoramic_analyses WHERE medical_record_id = ?i", $recordId);

// Get other radiographs for this record
$otherRadiographs = $db->getAll("SELECT * FROM other_radiographs WHERE medical_record_id = ?i", $recordId);

// Prepare data for cephalometric analysis table
$cephParameters = [
    'skeletal_horizontal' => [
        'sna' => [
            'name' => 'SNA (Steiner)',
            'mean' => '82°±2°',
            'value' => $cephalometricAnalysis['sna_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['sna_conclusion'] ?? ''
        ],
        'snb' => [
            'name' => 'SNB (Steiner)',
            'mean' => '80°±2°',
            'value' => $cephalometricAnalysis['snb_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['snb_conclusion'] ?? ''
        ],
        'anb' => [
            'name' => 'ANB (Steiner)',
            'mean' => '2°±2°',
            'value' => $cephalometricAnalysis['anb_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['anb_conclusion'] ?? ''
        ],
        'napg' => [
            'name' => 'NaPg (Downs)',
            'mean' => '0°±5,1°',
            'value' => $cephalometricAnalysis['napg_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['napg_conclusion'] ?? ''
        ]
    ],
    'skeletal_vertical' => [
        'mmpa' => [
            'name' => 'MMPA (Eastman)',
            'mean' => '27°±4°',
            'value' => $cephalometricAnalysis['mmpa_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['mmpa_conclusion'] ?? ''
        ],
        'fmpa' => [
            'name' => 'FMPA (Tweed)',
            'mean' => '25°±3°',
            'value' => $cephalometricAnalysis['fmpa_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['fmpa_conclusion'] ?? ''
        ]
    ],
    'dental' => [
        'fmia' => [
            'name' => 'FMIA',
            'mean' => '65°±2°',
            'value' => $cephalometricAnalysis['fmia_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['fmia_conclusion'] ?? ''
        ],
        'impa' => [
            'name' => 'IMPA',
            'mean' => '90°±5°',
            'value' => $cephalometricAnalysis['impa_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['impa_conclusion'] ?? ''
        ],
        'i_sn' => [
            'name' => 'I-SN',
            'mean' => '104°±6°',
            'value' => $cephalometricAnalysis['i_sn_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['i_sn_conclusion'] ?? ''
        ],
        'i_mxpi' => [
            'name' => 'I-MxPI',
            'mean' => '109°±6°',
            'value' => $cephalometricAnalysis['i_mxpi_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['i_mxpi_conclusion'] ?? ''
        ],
        'ui_na' => [
            'name' => 'Ui-NA',
            'mean' => '4mm±2mm',
            'value' => $cephalometricAnalysis['ui_na_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['ui_na_conclusion'] ?? ''
        ],
        'li_nb' => [
            'name' => 'Li-NB',
            'mean' => '4mm±2mm',
            'value' => $cephalometricAnalysis['li_nb_value'] ?? '',
            'conclusion' => $cephalometricAnalysis['li_nb_conclusion'] ?? ''
        ],
        'interincisal_angle' => [
            'name' => 'Interincisal Angle',
            'mean' => '135°±10°',
            'value' => $cephalometricAnalysis['interincisal_angle'] ?? '',
            'conclusion' => $cephalometricAnalysis['interincisal_angle_conclusion'] ?? ''
        ]
    ],
    'soft_tissue' => [
        'eline_upper_lip' => [
            'name' => 'E-Line Bibir Atas',
            'mean' => '1mm±2mm',
            'value' => $cephalometricAnalysis['e_line_upper_lip'] ?? '',
            'conclusion' => $cephalometricAnalysis['eline_upper_lip_conclusion'] ?? ''
        ],
        'eline_lower_lip' => [
            'name' => 'E-Line Bibir Bawah',
            'mean' => '0mm±2mm',
            'value' => $cephalometricAnalysis['e_line_lower_lip'] ?? '',
            'conclusion' => $cephalometricAnalysis['eline_lower_lip_conclusion'] ?? ''
        ]
    ]
];
?>

<div class="section-container">
    <h2 class="section-title">Analisis Radiologis</h2>

    <?php if (!empty($errorMessage)): ?>
    <div class="alert alert-danger"><?= $errorMessage ?></div>
    <?php endif; ?>

    <?php if (!empty($successMessage)): ?>
    <div class="alert alert-success"><?= $successMessage ?></div>
    <?php endif; ?>

    <!-- Cephalometric Analysis Section -->
    <div class="card">
        <div class="card-header">
            <h4>Analisis Sefalometri</h4>
        </div>
        <div class="card-body">
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Foto Sefalometri
                    Lateral</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="custom-file">
                        <input type="file" class="custom-file-input" id="cephalometric_image" name="cephalometric_image"
                            accept="image/*">
                        <label class="custom-file-label" for="cephalometric_image">Pilih file gambar</label>
                    </div>
                    <small class="form-text text-muted">Format: JPG, PNG, GIF. Ukuran maks: 2MB</small>

                    <?php if (!empty($cephalometricAnalysis['image_path'])): ?>
                    <div class="mt-3">
                        <p>Foto saat ini:</p>
                        <img src="<?= $cephalometricAnalysis['image_path'] ?>" alt="Foto Sefalometri Lateral"
                            class="img-thumbnail" style="max-height: 200px;">
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Cephalometric Canvas for Landmarks and Measurements -->
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Analisis Landmark</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="card">
                        <div class="card-body">
                            <?php if (empty($cephalometricAnalysis['image_path'])): ?>
                            <p class="text-muted">Fitur analisis landmark akan tersedia setelah unggah gambar
                                sefalometri dan menyimpan data.</p>
                            <?php else: ?>
                            <div class="mb-3">
                                <button type="button" class="btn btn-sm btn-primary" id="enable-landmark-mode">
                                    <i class="fas fa-edit"></i> Tandai Landmark
                                </button>
                                <button type="button" class="btn btn-sm btn-info" id="draw-lines">
                                    <i class="fas fa-draw-polygon"></i> Gambar Garis
                                </button>
                                <button type="button" class="btn btn-sm btn-warning" id="clear-landmarks">
                                    <i class="fas fa-trash"></i> Hapus Semua
                                </button>
                                <button type="button" class="btn btn-sm btn-success" id="save-landmarks">
                                    <i class="fas fa-save"></i> Simpan Landmark
                                </button>
                                <button type="button" class="btn btn-sm btn-secondary" id="calculate-measurements">
                                    <i class="fas fa-calculator"></i> Hitung Pengukuran
                                </button>
                            </div>
                            <?php endif; ?>

                            <!-- Placeholder for cephalometric canvas/drawing tool -->
                            <div id="cephalometric-canvas-container"
                                style="height: 700px; background-color: #f8f9fa; border: 1px dashed #ccc; <?= empty($cephalometricAnalysis['image_path']) ? 'display: flex; align-items: center; justify-content: center;' : '' ?>">
                                <?php if (empty($cephalometricAnalysis['image_path'])): ?>
                                <p class="text-center">Canvas untuk penandaan landmark sefalometri akan muncul di sini
                                </p>
                                <?php else: ?>
                                <canvas id="cephalometric-canvas" width="1000" height="700"></canvas>
                                <?php endif; ?>
                            </div>

                            <input type="hidden" name="landmarks_data" id="landmarks_data"
                                value="<?= htmlspecialchars($cephalometricAnalysis['landmark_data'] ?? '') ?>">

                            <div class="mt-3">
                                <p><strong>Landmark yang harus ditandai:</strong></p>
                                <div class="row">
                                    <div class="col-md-4">
                                        <ul>
                                            <li>Sella (S)</li>
                                            <li>Nasion (N)</li>
                                            <li>Orbitale (O)</li>
                                            <li>A</li>
                                            <li>B</li>
                                            <li>Pogonion (Pg)</li>
                                        </ul>
                                    </div>
                                    <div class="col-md-4">
                                        <ul>
                                            <li>Gonion (Go)</li>
                                            <li>Gnation (Gn)</li>
                                            <li>Menton (Me)</li>
                                            <li>Anterior Nasal Spine (ANS)</li>
                                            <li>Posterior Nasal Spine (PNS)</li>
                                            <li>Porion (Po)</li>
                                        </ul>
                                    </div>
                                    <div class="col-md-4">
                                        <ul>
                                            <li>Li, L0</li>
                                            <li>Ui, U0</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cephalometric Analysis Table -->
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Hasil Analisis
                    Sefalometri</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr class="bg-primary text-white">
                                    <th style="color: white;">Parameter</th>
                                    <th style="color: white;">Rata-Rata</th>
                                    <th style="color: white;">Pasien</th>
                                    <th style="color: white;">Kesimpulan</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Skeletal Parameters (Horizontal) -->
                                <tr class="bg-light">
                                    <td colspan="4"><strong>Nilai Parameter Skeletal (Horizontal)</strong></td>
                                </tr>
                                <?php foreach ($cephParameters['skeletal_horizontal'] as $key => $param): ?>
                                <tr>
                                    <td><?= $param['name'] ?></td>
                                    <td><?= $param['mean'] ?></td>
                                    <td>
                                        <input type="text" class="form-control" name="<?= $key ?>_value"
                                            value="<?= $param['value'] ?>" placeholder="0.0">
                                    </td>
                                    <td>
                                        <?php if ($key === 'sna'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Kedudukan Maksila terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Kedudukan Maksila terhadap basis cranii retrognati"
                                                    <?= strpos($param['conclusion'], 'retrognati') !== false ? 'selected' : '' ?>>
                                                    Retrognati</option>
                                                <option value="Kedudukan Maksila terhadap basis cranii orthognati"
                                                    <?= strpos($param['conclusion'], 'orthognati') !== false ? 'selected' : '' ?>>
                                                    Orthognati</option>
                                                <option value="Kedudukan Maksila terhadap basis cranii prognati"
                                                    <?= strpos($param['conclusion'], 'prognati') !== false ? 'selected' : '' ?>>
                                                    Prognati</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'snb'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Kedudukan Mandibula terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Kedudukan Mandibula terhadap basis cranii retrognati"
                                                    <?= strpos($param['conclusion'], 'retrognati') !== false ? 'selected' : '' ?>>
                                                    Retrognati</option>
                                                <option value="Kedudukan Mandibula terhadap basis cranii orthognati"
                                                    <?= strpos($param['conclusion'], 'orthognati') !== false ? 'selected' : '' ?>>
                                                    Orthognati</option>
                                                <option value="Kedudukan Mandibula terhadap basis cranii prognati"
                                                    <?= strpos($param['conclusion'], 'prognati') !== false ? 'selected' : '' ?>>
                                                    Prognati</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'anb'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Kedudukan mandibula terhadap
                                                    maksila</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option
                                                    value="Kedudukan mandibula terhadap maksila retrognati Pola Skeletal Kelas II"
                                                    <?= strpos($param['conclusion'], 'retrognati') !== false ? 'selected' : '' ?>>
                                                    Retrognati (Kelas II)</option>
                                                <option
                                                    value="Kedudukan mandibula terhadap maksila orthognati Pola Skeletal Kelas I"
                                                    <?= strpos($param['conclusion'], 'orthognati') !== false ? 'selected' : '' ?>>
                                                    Orthognati (Kelas I)</option>
                                                <option
                                                    value="Kedudukan mandibula terhadap maksila prognati Pola Skeletal Kelas III"
                                                    <?= strpos($param['conclusion'], 'prognati') !== false ? 'selected' : '' ?>>
                                                    Prognati (Kelas III)</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'napg'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Profil Skeletal</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Profil Skeletal cekung"
                                                    <?= strpos($param['conclusion'], 'cekung') !== false ? 'selected' : '' ?>>
                                                    Cekung</option>
                                                <option value="Profil Skeletal lurus"
                                                    <?= strpos($param['conclusion'], 'lurus') !== false ? 'selected' : '' ?>>
                                                    Lurus</option>
                                                <option value="Profil Skeletal cembung"
                                                    <?= strpos($param['conclusion'], 'cembung') !== false ? 'selected' : '' ?>>
                                                    Cembung</option>
                                            </select>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>

                                <!-- Skeletal Parameters (Vertical) -->
                                <tr class="bg-light">
                                    <td colspan="4"><strong>Nilai Parameter Skeletal (Vertikal)</strong></td>
                                </tr>
                                <?php foreach ($cephParameters['skeletal_vertical'] as $key => $param): ?>
                                <tr>
                                    <td><?= $param['name'] ?></td>
                                    <td><?= $param['mean'] ?></td>
                                    <td>
                                        <input type="text" class="form-control" name="<?= $key ?>_value"
                                            value="<?= $param['value'] ?>" placeholder="0.0">
                                    </td>
                                    <td>
                                        <?php if ($key === 'mmpa'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Kedudukan bidang maksila terhadap bidang
                                                    mandibula</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option
                                                    value="Kedudukan bidang maksila terhadap bidang mandibula retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option
                                                    value="Kedudukan bidang maksila terhadap bidang mandibula normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option
                                                    value="Kedudukan bidang maksila terhadap bidang mandibula protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'fmpa'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Pertumbuhan 1/3 muka bawah dalam arah
                                                    posteroinferior</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option
                                                    value="Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Kurang Dari Normal"
                                                    <?= strpos($param['conclusion'], 'Kurang') !== false ? 'selected' : '' ?>>
                                                    Kurang Dari Normal</option>
                                                <option
                                                    value="Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Normal"
                                                    <?= strpos($param['conclusion'], 'Normal') !== false && !strpos($param['conclusion'], 'Kurang') && !strpos($param['conclusion'], 'Lebih') ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option
                                                    value="Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Lebih Dari Normal"
                                                    <?= strpos($param['conclusion'], 'Lebih') !== false ? 'selected' : '' ?>>
                                                    Lebih Dari Normal</option>
                                            </select>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>

                                <!-- Dental Parameters -->
                                <tr class="bg-light">
                                    <td colspan="4"><strong>Nilai Parameter Dental</strong></td>
                                </tr>
                                <?php foreach ($cephParameters['dental'] as $key => $param): ?>
                                <tr>
                                    <td><?= $param['name'] ?></td>
                                    <td><?= $param['mean'] ?></td>
                                    <td>
                                        <input type="text" class="form-control" name="<?= $key ?>_value"
                                            value="<?= $param['value'] ?>" placeholder="0.0">
                                    </td>
                                    <td>
                                        <?php if ($key === 'fmia'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif bawah terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif bawah terhadap basis cranii retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif bawah terhadap basis cranii normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif bawah terhadap basis cranii protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'impa'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif bawah terhadap bidang
                                                    mandibula</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option
                                                    value="Inklinasi insisif bawah terhadap bidang mandibula retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif bawah terhadap bidang mandibula normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option
                                                    value="Inklinasi insisif bawah terhadap bidang mandibula protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'i_sn'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif atas terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif atas terhadap basis cranii retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif atas terhadap basis cranii normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif atas terhadap basis cranii protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'i_mxpi'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif atas terhadap bidang
                                                    maksila</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif atas terhadap bidang maksila retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif atas terhadap bidang maksila normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif atas terhadap bidang maksila protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'ui_na'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif atas terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif atas terhadap basis cranii retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif atas terhadap basis cranii normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif atas terhadap basis cranii protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'li_nb'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif bawah terhadap basis
                                                    cranii</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif bawah terhadap basis cranii retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif bawah terhadap basis cranii normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif bawah terhadap basis cranii protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'interincisal_angle'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Inklinasi insisif bawah terhadap insisif
                                                    atas</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Inklinasi insisif bawah terhadap insisif atas retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Inklinasi insisif bawah terhadap insisif atas normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Inklinasi insisif bawah terhadap insisif atas protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>

                                <!-- Soft Tissue Parameters -->
                                <tr class="bg-light">
                                    <td colspan="4"><strong>Nilai Parameter Jaringan Lunak</strong></td>
                                </tr>
                                <?php foreach ($cephParameters['soft_tissue'] as $key => $param): ?>
                                <tr>
                                    <td><?= $param['name'] ?></td>
                                    <td><?= $param['mean'] ?></td>
                                    <td>
                                        <input type="text" class="form-control" name="<?= $key ?>_value"
                                            value="<?= $param['value'] ?>" placeholder="0.0">
                                    </td>
                                    <td>
                                        <?php if ($key === 'eline_upper_lip'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Bibir atas</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Bibir atas retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Bibir atas normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Bibir atas protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php elseif ($key === 'eline_lower_lip'): ?>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text">Bibir bawah</span>
                                            </div>
                                            <select class="form-control" name="<?= $key ?>_conclusion">
                                                <option value="Bibir bawah retrusif"
                                                    <?= strpos($param['conclusion'], 'retrusif') !== false ? 'selected' : '' ?>>
                                                    Retrusif</option>
                                                <option value="Bibir bawah normal"
                                                    <?= strpos($param['conclusion'], 'normal') !== false ? 'selected' : '' ?>>
                                                    Normal</option>
                                                <option value="Bibir bawah protrusif"
                                                    <?= strpos($param['conclusion'], 'protrusif') !== false ? 'selected' : '' ?>>
                                                    Protrusif</option>
                                            </select>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Cephalometric Analysis Conclusion -->
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Kesimpulan Analisis
                    Sefalometri</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <textarea class="form-control" name="cephalometric_conclusion"
                        rows="4"><?= $cephalometricAnalysis['cephalometric_conclusion'] ?? '' ?></textarea>
                    <small class="form-text text-muted">Kesimpulan akan dihasilkan secara otomatis berdasarkan hasil
                        analisis di atas.</small>
                </div>
            </div>
        </div>
    </div>

    <!-- Panoramic Analysis Section -->
    <div class="card mt-4">
        <div class="card-header">
            <h4>Foto Panoramik</h4>
        </div>
        <div class="card-body">
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Foto Panoramik</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="custom-file">
                        <input type="file" class="custom-file-input" id="panoramic_image" name="panoramic_image"
                            accept="image/*">
                        <label class="custom-file-label" for="panoramic_image">Pilih file gambar</label>
                    </div>
                    <small class="form-text text-muted">Format: JPG, PNG, GIF. Ukuran maks: 2MB</small>

                    <?php if (!empty($panoramicAnalysis['image_path'])): ?>
                    <div class="mt-3">
                        <p>Foto saat ini:</p>
                        <img src="<?= $panoramicAnalysis['image_path'] ?>" alt="Foto Panoramik" class="img-thumbnail"
                            style="max-height: 200px;">
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Keadaan Tulang</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="bone_condition"
                        value="<?= $panoramicAnalysis['bone_condition'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Keadaan Jaringan
                    Periodontal</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="periodontal_condition"
                        value="<?= $panoramicAnalysis['periodontal_tissue'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Karies</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="caries"
                        value="<?= $panoramicAnalysis['caries'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Kehilangan Gigi</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="missing_teeth"
                        value="<?= $panoramicAnalysis['missing_teeth'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Agenesis</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="form-check form-check-inline">
                        <input class="form-check-input" type="radio" name="agenesis" id="agenesis_yes" value="Ada"
                            <?= isset($panoramicAnalysis['agenesis']) && $panoramicAnalysis['agenesis'] == 'Ada' ? 'checked' : '' ?>>
                        <label class="form-check-label" for="agenesis_yes">Ada</label>
                    </div>
                    <div class="form-check form-check-inline">
                        <input class="form-check-input" type="radio" name="agenesis" id="agenesis_no" value="Tidak Ada"
                            <?= isset($panoramicAnalysis['agenesis']) && $panoramicAnalysis['agenesis'] == 'Tidak Ada' ? 'checked' : '' ?>>
                        <label class="form-check-label" for="agenesis_no">Tidak Ada</label>
                    </div>

                    <div id="agenesis_details" class="mt-2"
                        style="<?= isset($panoramicAnalysis['agenesis']) && $panoramicAnalysis['agenesis'] == 'Ada' ? '' : 'display: none;' ?>">
                        <input type="text" class="form-control" name="agenesis_regio" placeholder="Regio..."
                            value="<?= $panoramicAnalysis['agenesis_region'] ?? '' ?>">
                    </div>
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Impaksi</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="impaction"
                        value="<?= $panoramicAnalysis['impaction'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Gigi Berlebih</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="supernumerary_teeth"
                        value="<?= $panoramicAnalysis['supernumerary_teeth'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Lain-Lain</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <input type="text" class="form-control" name="panoramic_other"
                        value="<?= $panoramicAnalysis['other_findings'] ?? '' ?>">
                </div>
            </div>

            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Urutan Erupsi Gigi</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="row">
                        <div class="col-md-6 mb-2">
                            <div class="card">
                                <div class="card-header py-1 text-center">
                                    <small>Kanan Atas</small>
                                </div>
                                <div class="card-body py-2">
                                    <input type="text" class="form-control" name="eruption_upper_right"
                                        value="<?= $panoramicAnalysis['eruption_sequence_top_right'] ?? '' ?>">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6 mb-2">
                            <div class="card">
                                <div class="card-header py-1 text-center">
                                    <small>Kiri Atas</small>
                                </div>
                                <div class="card-body py-2">
                                    <input type="text" class="form-control" name="eruption_upper_left"
                                        value="<?= $panoramicAnalysis['eruption_sequence_top_left'] ?? '' ?>">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header py-1 text-center">
                                    <small>Kanan Bawah</small>
                                </div>
                                <div class="card-body py-2">
                                    <input type="text" class="form-control" name="eruption_lower_right"
                                        value="<?= $panoramicAnalysis['eruption_sequence_bottom_right'] ?? '' ?>">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header py-1 text-center">
                                    <small>Kiri Bawah</small>
                                </div>
                                <div class="card-body py-2">
                                    <input type="text" class="form-control" name="eruption_lower_left"
                                        value="<?= $panoramicAnalysis['eruption_sequence_bottom_left'] ?? '' ?>">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Other Radiographs Section -->
    <div class="card mt-4">
        <div class="card-header">
            <h4>Lain-lain (Dental, CBCT, Carpal)</h4>
        </div>
        <div class="card-body">
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Radiograf Lain</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="custom-file">
                        <input type="file" class="custom-file-input" id="other_radiograph" name="other_radiograph[]"
                            accept="image/*,.pdf,.dcm" multiple>
                        <label class="custom-file-label" for="other_radiograph">Pilih file (bisa multiple)</label>
                    </div>
                    <small class="form-text text-muted">Format: JPG, PNG, PDF, DICOM. Ukuran maks: 5MB per
                        file</small>
                </div>
            </div>

            <?php if (!empty($otherRadiographs)): ?>
            <div class="form-group row mb-4">
                <label class="col-form-label sanitize col-12 col-md-3 col-lg-2">Radiograf yang Tersedia</label>
                <div class="col-sm-12 col-md-9 col-lg-10">
                    <div class="row">
                        <?php foreach ($otherRadiographs as $radiograph): ?>
                        <div class="col-md-4 mb-3">
                            <div class="card">
                                <div class="card-body text-center">
                                    <?php if (in_array(pathinfo($radiograph['image_path'], PATHINFO_EXTENSION), ['jpg', 'jpeg', 'png', 'gif'])): ?>
                                    <img src="<?= $radiograph['image_path'] ?>" class="img-fluid"
                                        style="max-height: 150px;">
                                    <?php else: ?>
                                    <i class="fas fa-file-medical fa-4x text-primary mb-2"></i>
                                    <p><?= basename($radiograph['image_path']) ?></p>
                                    <?php endif; ?>

                                    <div class="mt-2">
                                        <a href="<?= $radiograph['image_path'] ?>" class="btn btn-sm btn-info"
                                            target="_blank">
                                            <i class="fas fa-eye"></i> Lihat
                                        </a>
                                        <button type="button" class="btn btn-sm btn-danger delete-radiograph"
                                            data-id="<?= $radiograph['id'] ?>">
                                            <i class="fas fa-trash"></i> Hapus
                                        </button>
                                        <input type="hidden" name="delete_radiograph[]" class="delete-radiograph-input"
                                            id="delete_radiograph_<?= $radiograph['id'] ?>" value="">
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script src="../assets/modules/jquery.min.js"></script>
<!-- Add Fabric.js for canvas manipulation -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/fabric.js/5.3.1/fabric.min.js"></script>

<script>
$(document).ready(function() {
    // Custom file input
    $('.custom-file-input').on('change', function() {
        var fileName = $(this).val().split('\\').pop();
        $(this).next('.custom-file-label').addClass("selected").html(fileName);
    });

    // Handle agenesis radio change
    $('input[name="agenesis"]').change(function() {
        if ($(this).val() === 'Ada') {
            $('#agenesis_details').show();
        } else {
            $('#agenesis_details').hide();
        }
    });

    // Handle delete radiograph buttons
    $('.delete-radiograph').click(function() {
        var radiographId = $(this).data('id');
        if ($(this).hasClass('btn-danger')) {
            $('#delete_radiograph_' + radiographId).val(radiographId);
            $(this).removeClass('btn-danger').addClass('btn-secondary');
            $(this).html('<i class="fas fa-undo"></i> Batal Hapus');
        } else {
            $('#delete_radiograph_' + radiographId).val('');
            $(this).removeClass('btn-secondary').addClass('btn-danger');
            $(this).html('<i class="fas fa-trash"></i> Hapus');
        }
    });

    <?php if (!empty($cephalometricAnalysis['image_path'])): ?>
    // Initialize Fabric.js canvas
    var canvas = new fabric.Canvas('cephalometric-canvas', {
        selection: false, // Disable group selection
        defaultCursor: 'default',
        hoverCursor: 'default'
    });
    var canvasWidth = 1000;
    var canvasHeight = 700;
    var landmarkMode = false;
    var landmarks = {};
    var lines = [];

    // Load the cephalometric image
    fabric.Image.fromURL('<?= $cephalometricAnalysis['image_path'] ?>', function(img) {
        // Scale image to fit canvas while maintaining aspect ratio
        var scale = Math.min(canvasWidth / img.width, canvasHeight / img.height);
        img.scale(scale);

        // Center the image
        img.set({
            left: canvasWidth / 2,
            top: canvasHeight / 2,
            originX: 'center',
            originY: 'center',
            selectable: false,
            hoverCursor: 'default'
        });

        canvas.add(img);
        canvas.sendToBack(img);

        // Load existing landmarks if available
        loadLandmarks();
    });

    // Enable landmark mode
    $('#enable-landmark-mode').click(function() {
        landmarkMode = true;
        canvas.defaultCursor = 'crosshair';
        canvas.hoverCursor = 'crosshair';
        canvas.selection = false; // Disable selection box

        // Disable selection for all objects
        canvas.forEachObject(function(obj) {
            obj.selectable = false;
        });

        //while landmark mode, mark the cephalometric-canvas-container with a red border
        $('#cephalometric-canvas-container').css('border', '2px solid red');

        // Show a tooltip to guide the user
        alert(
            'Klik pada gambar untuk menandai landmark. Klik pada landmark untuk mengedit atau menghapus.'
        );
    });

    // Disable landmark mode when other buttons are clicked
    $('#draw-lines, #clear-landmarks, #save-landmarks').click(function() {
        landmarkMode = false;
        canvas.defaultCursor = 'default';
        canvas.hoverCursor = 'default';

        $('#cephalometric-canvas-container').css('border', '1px dashed #ccc');
    });

    // Draw lines between landmarks
    $('#draw-lines').click(function() {
        drawAllLines();
    });

    // Clear all landmarks
    $('#clear-landmarks').click(function() {
        if (confirm('Apakah Anda yakin ingin menghapus semua landmark?')) {
            clearLandmarks();
            clearLines();
            landmarks = {};
            saveLandmarks();
        }
    });

    // Save landmarks
    $('#save-landmarks').click(function() {
        saveLandmarks();
        calculateMeasurements();

        // Save landmarks via AJAX
        $.ajax({
            url: 'edit.php?id=<?= $recordId ?>&section=7',
            type: 'POST',
            data: {
                landmarks_data: $('#landmarks_data').val(),
                save_landmarks_only: '1'
            },
            success: function(response) {
                alert('Landmark berhasil disimpan!');
            },
            error: function() {
                alert('Gagal menyimpan landmark. Silakan coba lagi.');
            }
        });
    });

    // Calculate measurements
    $('#calculate-measurements').click(function() {
        calculateMeasurements();
        alert('Pengukuran sefalometri berhasil dihitung dan diperbarui.');
    });

    // Helper function to draw a line between two landmarks
    function drawLine(landmark1, landmark2, name) {
        var line = new fabric.Line([
            landmark1.left, landmark1.top,
            landmark2.left, landmark2.top
        ], {
            stroke: 'blue',
            strokeWidth: 2,
            selectable: false,
            hoverCursor: 'default',
            name: name
        });

        canvas.add(line);
        lines.push(line);

        // Add text label at the middle of the line
        var midX = (landmark1.left + landmark2.left) / 2;
        var midY = (landmark1.top + landmark2.top) / 2;

        var text = new fabric.Text(name, {
            left: midX,
            top: midY,
            fontSize: 12,
            fill: 'blue',
            backgroundColor: 'rgba(255,255,255,0.7)',
            originX: 'center',
            originY: 'center',
            selectable: false,
            hoverCursor: 'default',
            name: name + '_text'
        });

        canvas.add(text);
        lines.push(text);
    }

    // Helper function to clear all lines
    function clearLines() {
        for (var i = 0; i < lines.length; i++) {
            canvas.remove(lines[i]);
        }
        lines = [];
    }

    // Helper function to clear all landmarks
    function clearLandmarks() {
        for (var key in landmarks) {
            canvas.remove(landmarks[key]);
        }
    }

    // Helper function to save landmarks
    function saveLandmarks() {
        var data = {};
        for (var key in landmarks) {
            if (!key.endsWith('_text')) {
                data[key] = {
                    x: landmarks[key].left,
                    y: landmarks[key].top
                };
            }
        }

        $('#landmarks_data').val(JSON.stringify(data));
    }

    // Helper function to calculate angle between three points
    function calculateAngle(p1, p2, p3) {
        // Convert to radians
        var angle = Math.atan2(p3.y - p2.y, p3.x - p2.x) - Math.atan2(p1.y - p2.y, p1.x - p2.x);

        // Convert to degrees
        angle = angle * (180 / Math.PI);

        // Ensure angle is positive
        if (angle < 0) {
            angle += 360;
        }

        // Ensure angle is less than 180 degrees
        if (angle > 180) {
            angle = 360 - angle;
        }

        return angle; // Return the number, not a string
    }

    // Helper function to calculate perpendicular distance from a point to a line
    function calculatePerpendicularDistance(point, lineStart, lineEnd) {
        // Calculate line length
        var lineLength = Math.sqrt(Math.pow(lineEnd.x - lineStart.x, 2) + Math.pow(lineEnd.y - lineStart.y, 2));

        // Calculate distance
        var distance = Math.abs((lineEnd.y - lineStart.y) * point.x - (lineEnd.x - lineStart.x) * point.y +
            lineEnd.x * lineStart.y - lineEnd.y * lineStart.x) / lineLength;

        return distance; // Return the number, not a string
    }

    // Helper function to calculate angle between two lines
    function calculateAngleBetweenLines(line1Start, line1End, line2Start, line2End) {
        // Calculate direction vectors
        const vector1 = {
            x: line1End.x - line1Start.x,
            y: line1End.y - line1Start.y
        };

        const vector2 = {
            x: line2End.x - line2Start.x,
            y: line2End.y - line2Start.y
        };

        // Calculate dot product
        const dotProduct = vector1.x * vector2.x + vector1.y * vector2.y;

        // Calculate magnitudes
        const magnitude1 = Math.sqrt(vector1.x * vector1.x + vector1.y * vector1.y);
        const magnitude2 = Math.sqrt(vector2.x * vector2.x + vector2.y * vector2.y);

        // Calculate cosine of angle
        const cosAngle = dotProduct / (magnitude1 * magnitude2);

        // Calculate angle in radians and convert to degrees
        let angle = Math.acos(Math.min(Math.max(cosAngle, -1), 1)) * (180 / Math.PI);

        // Ensure angle is less than 180 degrees
        if (angle > 180) {
            angle = 360 - angle;
        }

        return angle;
    }

    // Function to calculate measurements from landmarks
    function calculateMeasurements() {
        // Check if we have enough landmarks
        if (!landmarks['S'] || !landmarks['N'] || !landmarks['A'] || !landmarks['B'] || !landmarks['Pg']) {
            alert(
                'Tidak cukup landmark untuk menghitung pengukuran. Pastikan minimal landmark S, N, A, B, dan Pg telah ditandai.'
            );
            return;
        }

        // Extract coordinates
        const S = {
            x: landmarks['S'].left,
            y: landmarks['S'].top
        };
        const N = {
            x: landmarks['N'].left,
            y: landmarks['N'].top
        };
        const A = {
            x: landmarks['A'].left,
            y: landmarks['A'].top
        };
        const B = {
            x: landmarks['B'].left,
            y: landmarks['B'].top
        };
        const Pg = {
            x: landmarks['Pg'].left,
            y: landmarks['Pg'].top
        };

        // Calculate SNA angle
        const snaAngle = calculateAngle(S, N, A);
        $('input[name="sna_value"]').val(snaAngle.toFixed(1));

        // Determine SNA conclusion
        let snaConclusion;
        if (snaAngle < 80) {
            snaConclusion = "Kedudukan Maksila terhadap basis cranii retrognati";
            $('select[name="sna_conclusion"]').val(snaConclusion);
        } else if (snaAngle <= 84) {
            snaConclusion = "Kedudukan Maksila terhadap basis cranii orthognati";
            $('select[name="sna_conclusion"]').val(snaConclusion);
        } else {
            snaConclusion = "Kedudukan Maksila terhadap basis cranii prognati";
            $('select[name="sna_conclusion"]').val(snaConclusion);
        }

        // Calculate SNB angle
        const snbAngle = calculateAngle(S, N, B);
        $('input[name="snb_value"]').val(snbAngle.toFixed(1));

        // Determine SNB conclusion
        let snbConclusion;
        if (snbAngle < 78) {
            snbConclusion = "Kedudukan Mandibula terhadap basis cranii retrognati";
            $('select[name="snb_conclusion"]').val(snbConclusion);
        } else if (snbAngle <= 82) {
            snbConclusion = "Kedudukan Mandibula terhadap basis cranii orthognati";
            $('select[name="snb_conclusion"]').val(snbConclusion);
        } else {
            snbConclusion = "Kedudukan Mandibula terhadap basis cranii prognati";
            $('select[name="snb_conclusion"]').val(snbConclusion);
        }

        // Calculate ANB angle
        const anbAngle = snaAngle - snbAngle;
        $('input[name="anb_value"]').val(anbAngle.toFixed(1));

        // Determine ANB conclusion
        let anbConclusion;
        if (anbAngle < 0) {
            anbConclusion = "Kedudukan mandibula terhadap maksila prognati Pola Skeletal Kelas III";
            $('select[name="anb_conclusion"]').val(anbConclusion);
        } else if (anbAngle <= 4) {
            anbConclusion = "Kedudukan mandibula terhadap maksila orthognati Pola Skeletal Kelas I";
            $('select[name="anb_conclusion"]').val(anbConclusion);
        } else {
            anbConclusion = "Kedudukan mandibula terhadap maksila retrognati Pola Skeletal Kelas II";
            $('select[name="anb_conclusion"]').val(anbConclusion);
        }

        // Calculate NaPg angle
        const napgAngle = 180 - calculateAngle(N, A, Pg);
        $('input[name="napg_value"]').val(napgAngle.toFixed(1));

        // Determine NaPg conclusion
        let napgConclusion;
        if (napgAngle < -5.1) {
            napgConclusion = "Profil Skeletal cekung";
            $('select[name="napg_conclusion"]').val(napgConclusion);
        } else if (napgAngle <= 5.1) {
            napgConclusion = "Profil Skeletal lurus";
            $('select[name="napg_conclusion"]').val(napgConclusion);
        } else {
            napgConclusion = "Profil Skeletal cembung";
            $('select[name="napg_conclusion"]').val(napgConclusion);
        }

        // Calculate more measurements if additional landmarks are available
        if (landmarks['ANS'] && landmarks['PNS'] && landmarks['Go'] && landmarks['Me']) {
            const ANS = {
                x: landmarks['ANS'].left,
                y: landmarks['ANS'].top
            };
            const PNS = {
                x: landmarks['PNS'].left,
                y: landmarks['PNS'].top
            };
            const Go = {
                x: landmarks['Go'].left,
                y: landmarks['Go'].top
            };
            const Me = {
                x: landmarks['Me'].left,
                y: landmarks['Me'].top
            };

            // Calculate MMPA (angle between maxillary and mandibular planes)
            const mmpaAngle = 180 - calculateAngleBetweenLines(ANS, PNS, Go, Me);
            $('input[name="mmpa_value"]').val(mmpaAngle.toFixed(1));

            // Determine MMPA conclusion
            let mmpaConclusion;
            if (mmpaAngle < 23) {
                mmpaConclusion = "Kedudukan bidang maksila terhadap bidang mandibula retrusif";
                $('select[name="mmpa_conclusion"]').val(mmpaConclusion);
            } else if (mmpaAngle <= 31) {
                mmpaConclusion = "Kedudukan bidang maksila terhadap bidang mandibula normal";
                $('select[name="mmpa_conclusion"]').val(mmpaConclusion);
            } else {
                mmpaConclusion = "Kedudukan bidang maksila terhadap bidang mandibula protrusif";
                $('select[name="mmpa_conclusion"]').val(mmpaConclusion);
            }
        }

        if (landmarks['Po'] && landmarks['O'] && landmarks['Go'] && landmarks['Me']) {
            const Po = {
                x: landmarks['Po'].left,
                y: landmarks['Po'].top
            };
            const O = {
                x: landmarks['O'].left,
                y: landmarks['O'].top
            };
            const Go = {
                x: landmarks['Go'].left,
                y: landmarks['Go'].top
            };
            const Me = {
                x: landmarks['Me'].left,
                y: landmarks['Me'].top
            };

            // Calculate FMPA (angle between Frankfurt horizontal and mandibular planes)
            const fmpaAngle = calculateAngleBetweenLines(Po, O, Go, Me);
            $('input[name="fmpa_value"]').val(fmpaAngle.toFixed(1));

            // Determine FMPA conclusion
            let fmpaConclusion;
            if (fmpaAngle < 22) {
                fmpaConclusion = "Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Kurang Dari Normal";
                $('select[name="fmpa_conclusion"]').val(fmpaConclusion);
            } else if (fmpaAngle <= 28) {
                fmpaConclusion = "Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Normal";
                $('select[name="fmpa_conclusion"]').val(fmpaConclusion);
            } else {
                fmpaConclusion = "Pertumbuhan 1/3 muka bawah dalam arah posteroinferior Lebih Dari Normal";
                $('select[name="fmpa_conclusion"]').val(fmpaConclusion);
            }
        }

        // Calculate dental parameters if landmarks are available
        if (landmarks['Po'] && landmarks['O'] && landmarks['Li'] && landmarks['L0']) {
            const Po = {
                x: landmarks['Po'].left,
                y: landmarks['Po'].top
            };
            const O = {
                x: landmarks['O'].left,
                y: landmarks['O'].top
            };
            const Li = {
                x: landmarks['Li'].left,
                y: landmarks['Li'].top
            };
            const L0 = {
                x: landmarks['L0'].left,
                y: landmarks['L0'].top
            };

            // Calculate FMIA (angle between Frankfurt horizontal and mandibular incisor)
            const fmiaAngle = calculateAngleBetweenLines(Po, O, Li, L0);
            $('input[name="fmia_value"]').val(fmiaAngle.toFixed(1));

            // Determine FMIA conclusion
            let fmiaConclusion;
            if (fmiaAngle < 63) {
                fmiaConclusion = "Inklinasi insisif bawah terhadap basis cranii protrusif";
                $('select[name="fmia_conclusion"]').val(fmiaConclusion);
            } else if (fmiaAngle <= 67) {
                fmiaConclusion = "Inklinasi insisif bawah terhadap basis cranii normal";
                $('select[name="fmia_conclusion"]').val(fmiaConclusion);
            } else {
                fmiaConclusion = "Inklinasi insisif bawah terhadap basis cranii retrusif";
                $('select[name="fmia_conclusion"]').val(fmiaConclusion);
            }
        }

        if (landmarks['Go'] && landmarks['Me'] && landmarks['Li'] && landmarks['L0']) {
            const Go = {
                x: landmarks['Go'].left,
                y: landmarks['Go'].top
            };
            const Me = {
                x: landmarks['Me'].left,
                y: landmarks['Me'].top
            };
            const Li = {
                x: landmarks['Li'].left,
                y: landmarks['Li'].top
            };
            const L0 = {
                x: landmarks['L0'].left,
                y: landmarks['L0'].top
            };

            // Calculate IMPA (angle between mandibular plane and mandibular incisor)
            const impaAngle = calculateAngleBetweenLines(Go, Me, Li, L0);
            $('input[name="impa_value"]').val(impaAngle.toFixed(1));

            // Determine IMPA conclusion
            let impaConclusion;
            if (impaAngle < 85) {
                impaConclusion = "Inklinasi insisif bawah terhadap bidang mandibula retrusif";
                $('select[name="impa_conclusion"]').val(impaConclusion);
            } else if (impaAngle <= 95) {
                impaConclusion = "Inklinasi insisif bawah terhadap bidang mandibula normal";
                $('select[name="impa_conclusion"]').val(impaConclusion);
            } else {
                impaConclusion = "Inklinasi insisif bawah terhadap bidang mandibula protrusif";
                $('select[name="impa_conclusion"]').val(impaConclusion);
            }
        }

        if (landmarks['S'] && landmarks['N'] && landmarks['Ui'] && landmarks['U0']) {
            const S = {
                x: landmarks['S'].left,
                y: landmarks['S'].top
            };
            const N = {
                x: landmarks['N'].left,
                y: landmarks['N'].top
            };
            const Ui = {
                x: landmarks['Ui'].left,
                y: landmarks['Ui'].top
            };
            const U0 = {
                x: landmarks['U0'].left,
                y: landmarks['U0'].top
            };

            // Calculate I-SN (angle between SN line and maxillary incisor)
            const isnAngle = calculateAngleBetweenLines(S, N, Ui, U0);
            $('input[name="i_sn_value"]').val(isnAngle.toFixed(1));

            // Determine I-SN conclusion
            let isnConclusion;
            if (isnAngle < 98) {
                isnConclusion = "Inklinasi insisif atas terhadap basis cranii retrusif";
                $('select[name="i_sn_conclusion"]').val(isnConclusion);
            } else if (isnAngle <= 110) {
                isnConclusion = "Inklinasi insisif atas terhadap basis cranii normal";
                $('select[name="i_sn_conclusion"]').val(isnConclusion);
            } else {
                isnConclusion = "Inklinasi insisif atas terhadap basis cranii protrusif";
                $('select[name="i_sn_conclusion"]').val(isnConclusion);
            }
        }

        if (landmarks['ANS'] && landmarks['PNS'] && landmarks['Ui'] && landmarks['U0']) {
            const ANS = {
                x: landmarks['ANS'].left,
                y: landmarks['ANS'].top
            };
            const PNS = {
                x: landmarks['PNS'].left,
                y: landmarks['PNS'].top
            };
            const Ui = {
                x: landmarks['Ui'].left,
                y: landmarks['Ui'].top
            };
            const U0 = {
                x: landmarks['U0'].left,
                y: landmarks['U0'].top
            };

            // Calculate I-MxPI (angle between maxillary plane and maxillary incisor)
            const imxpiAngle = calculateAngleBetweenLines(ANS, PNS, Ui, U0);
            $('input[name="i_mxpi_value"]').val(imxpiAngle.toFixed(1));

            // Determine I-MxPI conclusion
            let imxpiConclusion;
            if (imxpiAngle < 103) {
                imxpiConclusion = "Inklinasi insisif atas terhadap bidang maksila retrusif";
                $('select[name="i_mxpi_conclusion"]').val(imxpiConclusion);
            } else if (imxpiAngle <= 115) {
                imxpiConclusion = "Inklinasi insisif atas terhadap bidang maksila normal";
                $('select[name="i_mxpi_conclusion"]').val(imxpiConclusion);
            } else {
                imxpiConclusion = "Inklinasi insisif atas terhadap bidang maksila protrusif";
                $('select[name="i_mxpi_conclusion"]').val(imxpiConclusion);
            }
        }

        if (landmarks['N'] && landmarks['A'] && landmarks['Ui']) {
            const N = {
                x: landmarks['N'].left,
                y: landmarks['N'].top
            };
            const A = {
                x: landmarks['A'].left,
                y: landmarks['A'].top
            };
            const Ui = {
                x: landmarks['Ui'].left,
                y: landmarks['Ui'].top
            };

            // Calculate Ui-NA (distance from upper incisor to NA line)
            const uinaDistance = calculatePerpendicularDistance(Ui, N, A);
            $('input[name="ui_na_value"]').val(uinaDistance.toFixed(1));

            // Determine Ui-NA conclusion
            let uinaConclusion;
            if (uinaDistance < 2) {
                uinaConclusion = "Inklinasi insisif atas terhadap basis cranii retrusif";
                $('select[name="ui_na_conclusion"]').val(uinaConclusion);
            } else if (uinaDistance <= 6) {
                uinaConclusion = "Inklinasi insisif atas terhadap basis cranii normal";
                $('select[name="ui_na_conclusion"]').val(uinaConclusion);
            } else {
                uinaConclusion = "Inklinasi insisif atas terhadap basis cranii protrusif";
                $('select[name="ui_na_conclusion"]').val(uinaConclusion);
            }
        }

        if (landmarks['N'] && landmarks['B'] && landmarks['Li']) {
            const N = {
                x: landmarks['N'].left,
                y: landmarks['N'].top
            };
            const B = {
                x: landmarks['B'].left,
                y: landmarks['B'].top
            };
            const Li = {
                x: landmarks['Li'].left,
                y: landmarks['Li'].top
            };

            // Calculate Li-NB (distance from lower incisor to NB line)
            const linbDistance = calculatePerpendicularDistance(Li, N, B);
            $('input[name="li_nb_value"]').val(linbDistance.toFixed(1));

            // Determine Li-NB conclusion
            let linbConclusion;
            if (linbDistance < 2) {
                linbConclusion = "Inklinasi insisif bawah terhadap basis cranii retrusif";
                $('select[name="li_nb_conclusion"]').val(linbConclusion);
            } else if (linbDistance <= 6) {
                linbConclusion = "Inklinasi insisif bawah terhadap basis cranii normal";
                $('select[name="li_nb_conclusion"]').val(linbConclusion);
            } else {
                linbConclusion = "Inklinasi insisif bawah terhadap basis cranii protrusif";
                $('select[name="li_nb_conclusion"]').val(linbConclusion);
            }
        }

        if (landmarks['Ui'] && landmarks['U0'] && landmarks['Li'] && landmarks['L0']) {
            const Ui = {
                x: landmarks['Ui'].left,
                y: landmarks['Ui'].top
            };
            const U0 = {
                x: landmarks['U0'].left,
                y: landmarks['U0'].top
            };
            const Li = {
                x: landmarks['Li'].left,
                y: landmarks['Li'].top
            };
            const L0 = {
                x: landmarks['L0'].left,
                y: landmarks['L0'].top
            };

            // Calculate interincisal angle
            const interincisalAngle = calculateAngleBetweenLines(Ui, U0, Li, L0);
            $('input[name="interincisal_angle_value"]').val(interincisalAngle.toFixed(1));

            // Determine interincisal angle conclusion
            let interincisalConclusion;
            if (interincisalAngle < 125) {
                interincisalConclusion = "Inklinasi insisif bawah terhadap insisif atas protrusif";
                $('select[name="interincisal_angle_conclusion"]').val(interincisalConclusion);
            } else if (interincisalAngle <= 145) {
                interincisalConclusion = "Inklinasi insisif bawah terhadap insisif atas normal";
                $('select[name="interincisal_angle_conclusion"]').val(interincisalConclusion);
            } else {
                interincisalConclusion = "Inklinasi insisif bawah terhadap insisif atas retrusif";
                $('select[name="interincisal_angle_conclusion"]').val(interincisalConclusion);
            }
        }

        // Soft tissue parameters
        if (landmarks['UL'] && landmarks['LL']) {
            const UL = {
                x: landmarks['UL'].left,
                y: landmarks['UL'].top
            };
            const LL = {
                x: landmarks['LL'].left,
                y: landmarks['LL'].top
            };

            // E-Line measurements would require additional landmarks for the E-Line itself
            // For now, we'll use placeholder values
            const elineUpperLip = 1.5; // Placeholder
            $('input[name="eline_upper_lip_value"]').val(elineUpperLip.toFixed(1));

            // Determine E-Line upper lip conclusion
            let elineUpperLipConclusion;
            if (elineUpperLip < -1) {
                elineUpperLipConclusion = "Bibir atas retrusif";
                $('select[name="eline_upper_lip_conclusion"]').val(elineUpperLipConclusion);
            } else if (elineUpperLip <= 3) {
                elineUpperLipConclusion = "Bibir atas normal";
                $('select[name="eline_upper_lip_conclusion"]').val(elineUpperLipConclusion);
            } else {
                elineUpperLipConclusion = "Bibir atas protrusif";
                $('select[name="eline_upper_lip_conclusion"]').val(elineUpperLipConclusion);
            }

            const elineLowerLip = 0.5; // Placeholder
            $('input[name="eline_lower_lip_value"]').val(elineLowerLip.toFixed(1));

            // Determine E-Line lower lip conclusion
            let elineLowerLipConclusion;
            if (elineLowerLip < -2) {
                elineLowerLipConclusion = "Bibir bawah retrusif";
                $('select[name="eline_lower_lip_conclusion"]').val(elineLowerLipConclusion);
            } else if (elineLowerLip <= 2) {
                elineLowerLipConclusion = "Bibir bawah normal";
                $('select[name="eline_lower_lip_conclusion"]').val(elineLowerLipConclusion);
            } else {
                elineLowerLipConclusion = "Bibir bawah protrusif";
                $('select[name="eline_lower_lip_conclusion"]').val(elineLowerLipConclusion);
            }
        }

        // Generate comprehensive conclusion
        const conclusion = generateCephalometricConclusion();
        $('textarea[name="cephalometric_conclusion"]').val(conclusion);

    }

    // Function to generate comprehensive cephalometric conclusion
    function generateCephalometricConclusion() {
        // Get values from form fields
        const snaConclusion = $('select[name="sna_conclusion"]').val() || '';
        const snbConclusion = $('select[name="snb_conclusion"]').val() || '';
        const anbConclusion = $('select[name="anb_conclusion"]').val() || '';
        const napgConclusion = $('select[name="napg_conclusion"]').val() || '';
        const fmpaConclusion = $('select[name="fmpa_conclusion"]').val() || '';
        const isnConclusion = $('select[name="i_sn_conclusion"]').val() || '';
        const imxpiConclusion = $('select[name="i_mxpi_conclusion"]').val() || '';
        const linbConclusion = $('select[name="li_nb_conclusion"]').val() || '';
        const impaConclusion = $('select[name="impa_conclusion"]').val() || '';
        const interincisalConclusion = $('select[name="interincisal_angle_conclusion"]').val() || '';
        const elineUpperLipConclusion = $('select[name="eline_upper_lip_conclusion"]').val() || '';
        const elineLowerLipConclusion = $('select[name="eline_lower_lip_conclusion"]').val() || '';

        // Extract skeletal class from ANB conclusion
        let skeletalClass = "I";
        if (anbConclusion && anbConclusion.includes("Kelas II")) {
            skeletalClass = "II";
        } else if (anbConclusion && anbConclusion.includes("Kelas III")) {
            skeletalClass = "III";
        }

        // Extract maxilla position from SNA conclusion
        let maxillaPosition = "orthognati";
        if (snaConclusion && snaConclusion.includes("retrognati")) {
            maxillaPosition = "retrognati";
        } else if (snaConclusion && snaConclusion.includes("prognati")) {
            maxillaPosition = "prognati";
        }

        // Extract mandible position from SNB conclusion
        let mandiblePosition = "orthognati";
        if (snbConclusion && snbConclusion.includes("retrognati")) {
            mandiblePosition = "retrognati";
        } else if (snbConclusion && snbConclusion.includes("prognati")) {
            mandiblePosition = "prognati";
        }

        // Extract skeletal profile from NaPg conclusion
        let skeletalProfile = "lurus";
        if (napgConclusion && napgConclusion.includes("cekung")) {
            skeletalProfile = "cekung";
        } else if (napgConclusion && napgConclusion.includes("cembung")) {
            skeletalProfile = "cembung";
        }

        // Extract vertical growth from FMPA conclusion
        let verticalGrowth = "Normal";
        if (fmpaConclusion && fmpaConclusion.includes("Kurang")) {
            verticalGrowth = "Kurang Dari Normal";
        } else if (fmpaConclusion && fmpaConclusion.includes("Lebih")) {
            verticalGrowth = "Lebih Dari Normal";
        }

        // Extract upper incisor inclination to cranial base from I-SN conclusion
        let upperIncToCranialBase = "normal";
        if (isnConclusion && isnConclusion.includes("retrusif")) {
            upperIncToCranialBase = "retrusif";
        } else if (isnConclusion && isnConclusion.includes("protrusif")) {
            upperIncToCranialBase = "protrusif";
        }

        // Extract upper incisor inclination to maxillary plane from I-MxPI conclusion
        let upperIncToMaxPlane = "normal";
        if (imxpiConclusion && imxpiConclusion.includes("retrusif")) {
            upperIncToMaxPlane = "retrusif";
        } else if (imxpiConclusion && imxpiConclusion.includes("protrusif")) {
            upperIncToMaxPlane = "protrusif";
        }

        // Extract lower incisor inclination to cranial base from Li-NB conclusion
        let lowerIncToCranialBase = "normal";
        if (linbConclusion && linbConclusion.includes("retrusif")) {
            lowerIncToCranialBase = "retrusif";
        } else if (linbConclusion && linbConclusion.includes("protrusif")) {
            lowerIncToCranialBase = "protrusif";
        }

        // Extract lower incisor inclination to mandibular plane from IMPA conclusion
        let lowerIncToMandPlane = "normal";
        if (impaConclusion && impaConclusion.includes("retrusif")) {
            lowerIncToMandPlane = "retrusif";
        } else if (impaConclusion && impaConclusion.includes("protrusif")) {
            lowerIncToMandPlane = "protrusif";
        }

        // Extract interincisal relationship from interincisal angle conclusion
        let interincisalRel = "normal";
        if (interincisalConclusion && interincisalConclusion.includes("retrusif")) {
            interincisalRel = "retrusif";
        } else if (interincisalConclusion && interincisalConclusion.includes("protrusif")) {
            interincisalRel = "protrusif";
        }

        // Extract upper lip position from E-Line conclusion
        let upperLipPosition = "normal";
        if (elineUpperLipConclusion && elineUpperLipConclusion.includes("retrusif")) {
            upperLipPosition = "retrusif";
        } else if (elineUpperLipConclusion && elineUpperLipConclusion.includes("protrusif")) {
            upperLipPosition = "protrusif";
        }

        // Extract lower lip position from E-Line conclusion
        let lowerLipPosition = "normal";
        if (elineLowerLipConclusion && elineLowerLipConclusion.includes("retrusif")) {
            lowerLipPosition = "retrusif";
        } else if (elineLowerLipConclusion && elineLowerLipConclusion.includes("protrusif")) {
            lowerLipPosition = "protrusif";
        }

        // Construct the comprehensive conclusion
        return `Maloklusi skeletal kelas ${skeletalClass} (Hubungan maksila dan mandibula ${anbConclusion ? (anbConclusion.includes("retrognati") ? "retrognati" : anbConclusion.includes("prognati") ? "prognati" : "orthognati") : "orthognati"}) dengan maksila ${maxillaPosition} terhadap basis kranii dan mandibula ${mandiblePosition} terhadap basis kranii. Profil skeletal ${skeletalProfile} dan pertumbuhan vertikal ⅓ wajah bawah ${verticalGrowth}. Inklinasi insisif atas terhadap basis kranii ${upperIncToCranialBase} dan inklinasi insisif atas terhadap bidang maksila ${upperIncToMaxPlane}, inklinasi insisif bawah terhadap basis kranii ${lowerIncToCranialBase} dan inklinasi insisif bawah terhadap bidang mandibula ${lowerIncToMandPlane}, serta inklinasi insisif bawah terhadap atas ${interincisalRel}. Posisi bibir atas ${upperLipPosition}. Posisi bibir bawah ${lowerLipPosition}.`;
    }

    // Helper function to load landmarks
    function loadLandmarks() {
        var landmarksData = $('#landmarks_data').val();
        if (!landmarksData) return;

        try {
            var data = JSON.parse(landmarksData);

            for (var key in data) {
                var x = data[key].x;
                var y = data[key].y;

                // Create a landmark
                var landmark = new fabric.Circle({
                    left: x,
                    top: y,
                    radius: 5,
                    fill: 'red',
                    stroke: 'white',
                    strokeWidth: 1,
                    originX: 'center',
                    originY: 'center',
                    hasControls: false,
                    hasBorders: false,
                    name: key,
                    type: 'circle'
                });

                // Add text label
                var text = new fabric.Text(key, {
                    left: x + 10,
                    top: y - 10,
                    fontSize: 12,
                    fill: 'red',
                    originX: 'left',
                    originY: 'bottom',
                    selectable: false,
                    name: key + '_text'
                });

                canvas.add(landmark);
                canvas.add(text);

                // Store the landmark
                landmarks[key] = landmark;
                landmarks[key + '_text'] = text;
            }

            canvas.renderAll();

            // Automatically draw lines after loading landmarks
            drawAllLines();
        } catch (e) {
            console.error('Error loading landmarks:', e);
        }
    }

    // Helper function to draw all lines between landmarks
    function drawAllLines() {
        // Clear existing lines
        clearLines();

        // Define the lines to draw based on standard cephalometric analysis
        if (landmarks['S'] && landmarks['N']) {
            drawLine(landmarks['S'], landmarks['N'], 'SN Plane');
        }

        if (landmarks['Po'] && landmarks['O']) {
            drawLine(landmarks['Po'], landmarks['O'], 'Frankfurt Horizontal');
        }

        if (landmarks['ANS'] && landmarks['PNS']) {
            drawLine(landmarks['ANS'], landmarks['PNS'], 'Palatal Plane');
        }

        if (landmarks['Go'] && landmarks['Me']) {
            drawLine(landmarks['Go'], landmarks['Me'], 'Mandibular Plane');
        }

        if (landmarks['N'] && landmarks['A']) {
            drawLine(landmarks['N'], landmarks['A'], 'NA Line');
        }

        if (landmarks['N'] && landmarks['B']) {
            drawLine(landmarks['N'], landmarks['B'], 'NB Line');
        }

        if (landmarks['N'] && landmarks['Pg']) {
            drawLine(landmarks['N'], landmarks['Pg'], 'NPg Line');
        }

        canvas.renderAll();
    }

    // Add canvas click handler for adding landmarks
    canvas.on('mouse:down', function(options) {
        if (!landmarkMode) return;

        // Get click coordinates
        var pointer = canvas.getPointer(options.e);
        var x = pointer.x;
        var y = pointer.y;

        // Check if clicked on an existing landmark
        var clickedOnLandmark = false;
        for (var key in landmarks) {
            if (!key.endsWith('_text')) {
                var landmark = landmarks[key];
                var dx = landmark.left - x;
                var dy = landmark.top - y;
                var distance = Math.sqrt(dx * dx + dy * dy);

                if (distance <= 10) {
                    // Clicked on an existing landmark
                    clickedOnLandmark = true;

                    if (confirm('Hapus landmark ' + key + '?')) {
                        // Remove the landmark and its label
                        canvas.remove(landmarks[key]);
                        canvas.remove(landmarks[key + '_text']);

                        // Remove from landmarks object
                        delete landmarks[key];
                        delete landmarks[key + '_text'];

                        // Save the updated landmarks
                        saveLandmarks();
                    }

                    break;
                }
            }
        }

        if (!clickedOnLandmark) {
            // Prompt for landmark name
            var landmarkName = prompt(
                'Masukkan nama landmark (S, N, O, A, B, Pg, Go, Gn, Me, ANS, PNS, Po, Ls, Li, L0, Ui, U0, UL, LL):'
            );

            if (landmarkName) {
                // Validate landmark name against the list of valid landmarks
                var validLandmarks = ['S', 'N', 'O', 'A', 'B', 'Pg', 'Go', 'Gn', 'Me', 'ANS', 'PNS',
                    'Po', 'Ls', 'Li', 'L0', 'Ui', 'U0', 'UL', 'LL'
                ];

                if (validLandmarks.includes(landmarkName)) {
                    // Remove existing landmark with the same name if it exists
                    if (landmarks[landmarkName]) {
                        canvas.remove(landmarks[landmarkName]);
                        canvas.remove(landmarks[landmarkName + '_text']);
                    }

                    // Create a new landmark
                    var landmark = new fabric.Circle({
                        left: x,
                        top: y,
                        radius: 5,
                        fill: 'red',
                        stroke: 'white',
                        strokeWidth: 1,
                        originX: 'center',
                        originY: 'center',
                        hasControls: false,
                        hasBorders: false,
                        selectable: false,
                        hoverCursor: 'pointer',
                        name: landmarkName,
                        type: 'circle'
                    });

                    // Add text label
                    var text = new fabric.Text(landmarkName, {
                        left: x + 10,
                        top: y - 10,
                        fontSize: 12,
                        fill: 'red',
                        originX: 'left',
                        originY: 'bottom',
                        selectable: false,
                        hoverCursor: 'default',
                        name: landmarkName + '_text'
                    });

                    canvas.add(landmark);
                    canvas.add(text);

                    // Store the landmark
                    landmarks[landmarkName] = landmark;
                    landmarks[landmarkName + '_text'] = text;

                    // Save the landmarks
                    saveLandmarks();
                } else {
                    alert(
                        'Nama landmark tidak valid. Gunakan salah satu dari: S, N, O, A, B, Pg, Go, Gn, Me, ANS, PNS, Po, Ls, Li, L0, Ui, U0, UL, LL'
                    );
                }
            }
        }
    });
    <?php endif; ?>
});
</script>