<?php
// Include initialization file
require_once '../init.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}

// Check if ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    setFlashMessage('ID pasien tidak valid.', 'danger');
    redirect('index.php');
}

$patientId = intval($_GET['id']);

// Get patient data
$patient = $db->getRow("SELECT 
    p.*,
    TIMESTAMPDIFF(YEAR, p.birth_date, CURDATE()) as age,
    creator.name as created_by_name,
    updater.name as updated_by_name
    FROM patients p
    LEFT JOIN users creator ON p.created_by = creator.id
    LEFT JOIN users updater ON p.updated_by = updater.id
    WHERE p.id = ?i", $patientId);

if (!$patient) {
    setFlashMessage('Pasien tidak ditemukan.', 'danger');
    redirect('index.php');
}

// Get medical records for this patient
$medicalRecords = $db->getAll("SELECT 
    mr.*,
    u.name as doctor_name
    FROM medical_records mr
    LEFT JOIN users u ON mr.doctor_id = u.id
    WHERE mr.patient_id = ?i
    ORDER BY mr.examination_date DESC", $patientId);
?>
<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta content="width=device-width, initial-scale=1, maximum-scale=1, shrink-to-fit=no" name="viewport">
    <title>Detail Pasien &mdash; Sistem Rekam Medis Ortodontik</title>

    <!-- General CSS Files -->
    <link rel="stylesheet" href="../assets/modules/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="../assets/modules/fontawesome/css/all.min.css">

    <!-- CSS Libraries -->
    <link rel="stylesheet" href="../assets/modules/datatables/datatables.min.css">
    <link rel="stylesheet" href="../assets/modules/datatables/DataTables-1.10.16/css/dataTables.bootstrap4.min.css">
    <link rel="stylesheet" href="../assets/modules/datatables/Select-1.2.4/css/select.bootstrap4.min.css">

    <!-- Template CSS -->
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/components.css">
</head>

<body>
    <div id="app">
        <div class="main-wrapper main-wrapper-1">
            <!-- Header -->
            <?php include '../layouts/header.php'; ?>

            <!-- Sidebar -->
            <?php include '../layouts/sidebar.php'; ?>

            <!-- Main Content -->
            <div class="main-content">
                <section class="section">
                    <div class="section-header">
                        <h1>Detail Pasien</h1>
                        <div class="section-header-breadcrumb">
                            <div class="breadcrumb-item active"><a href="../index.php">Dashboard</a></div>
                            <div class="breadcrumb-item"><a href="index.php">Data Pasien</a></div>
                            <div class="breadcrumb-item">Detail Pasien</div>
                        </div>
                    </div>

                    <div class="section-body">
                        <h2 class="section-title">Informasi Pasien: <?php echo $patient['name']; ?></h2>
                        <p class="section-lead">
                            Detail informasi data pasien.
                        </p>

                        <div class="row">
                            <div class="col-12">
                                <div class="card">
                                    <div class="card-header">
                                        <h4>Data Diri Pasien</h4>
                                        <div class="card-header-action">
                                            <a href="edit.php?id=<?php echo $patient['id']; ?>" class="btn btn-primary">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <table class="table table-striped">
                                                    <tr>
                                                        <th>Nomor Rekam Medis</th>
                                                        <td><?php echo $patient['medical_record_number']; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Nama Lengkap</th>
                                                        <td><?php echo $patient['name']; ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Tanggal Lahir</th>
                                                        <td><?php echo formatDate($patient['birth_date']); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Usia</th>
                                                        <td><?php echo $patient['age']; ?> tahun</td>
                                                    </tr>
                                                    <tr>
                                                        <th>Jenis Kelamin</th>
                                                        <td><?php echo $patient['gender']; ?></td>
                                                    </tr>
                                                </table>
                                            </div>
                                            <div class="col-md-6">
                                                <table class="table table-striped">
                                                    <tr>
                                                        <th>Alamat</th>
                                                        <td><?php echo !empty($patient['address']) ? $patient['address'] : '-'; ?>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <th>Nomor Telepon</th>
                                                        <td><?php echo !empty($patient['phone_number']) ? $patient['phone_number'] : '-'; ?>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <th>Email</th>
                                                        <td><?php echo !empty($patient['email']) ? $patient['email'] : '-'; ?>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <th>Pekerjaan</th>
                                                        <td><?php echo !empty($patient['occupation']) ? $patient['occupation'] : '-'; ?>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </div>

                                        <div class="row mt-4">
                                            <div class="col-md-12">
                                                <div class="section-title mt-0">Informasi Tambahan</div>
                                                <table class="table table-striped">
                                                    <tr>
                                                        <th width="200">Dibuat Pada</th>
                                                        <td><?php echo formatDateTime($patient['created_at']); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Dibuat Oleh</th>
                                                        <td><?php echo $patient['created_by_name']; ?></td>
                                                    </tr>
                                                    <?php if (!empty($patient['updated_at'])): ?>
                                                    <tr>
                                                        <th>Diperbarui Pada</th>
                                                        <td><?php echo formatDateTime($patient['updated_at']); ?></td>
                                                    </tr>
                                                    <tr>
                                                        <th>Diperbarui Oleh</th>
                                                        <td><?php echo $patient['updated_by_name']; ?></td>
                                                    </tr>
                                                    <?php endif; ?>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12">
                                <div class="card">
                                    <div class="card-header">
                                        <h4>Riwayat Rekam Medis</h4>
                                        <?php if (isDoctor()): ?>
                                        <div class="card-header-action">
                                            <a href="../medical_records/create.php?patient_id=<?php echo $patient['id']; ?>"
                                                class="btn btn-primary">
                                                <i class="fas fa-plus"></i> Tambah Rekam Medis
                                            </a>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="card-body">
                                        <?php if (empty($medicalRecords)): ?>
                                        <div class="empty-state" data-height="400">
                                            <div class="empty-state-icon">
                                                <i class="fas fa-file-medical"></i>
                                            </div>
                                            <h2>Belum ada rekam medis</h2>
                                            <p class="lead">
                                                Pasien ini belum memiliki rekam medis.
                                                <?php if (isDoctor()): ?>
                                                Klik tombol "Tambah Rekam Medis" untuk membuat rekam medis baru.
                                                <?php endif; ?>
                                            </p>
                                        </div>
                                        <?php else: ?>
                                        <div class="table-responsive">
                                            <table class="table table-striped" id="medical-records-table">
                                                <thead>
                                                    <tr>
                                                        <th>No</th>
                                                        <th>Tanggal Pemeriksaan</th>
                                                        <th>Dokter</th>
                                                        <th>Status</th>
                                                        <th>Aksi</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php $no = 1; foreach ($medicalRecords as $record): ?>
                                                    <tr>
                                                        <td><?php echo $no++; ?></td>
                                                        <td><?php echo formatDate($record['examination_date']); ?></td>
                                                        <td><?php echo $record['doctor_name']; ?></td>
                                                        <td>
                                                            <?php if ($record['is_complete'] == 1): ?>
                                                            <span class="badge badge-success">Selesai</span>
                                                            <?php else: ?>
                                                            <span class="badge badge-warning">Dalam Proses</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <a href="../medical_records/view.php?id=<?php echo $record['id']; ?>"
                                                                class="btn btn-info btn-sm">
                                                                <i class="fas fa-eye"></i> Lihat
                                                            </a>
                                                            <?php if (isDoctor() && $record['is_complete'] == 0): ?>
                                                            <a href="../medical_records/edit.php?id=<?php echo $record['id']; ?>"
                                                                class="btn btn-primary btn-sm">
                                                                <i class="fas fa-edit"></i> Edit
                                                            </a>
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            </div>

            <!-- Footer -->
            <?php include '../layouts/footer.php'; ?>
        </div>
    </div>

    <!-- General JS Scripts -->
    <script src="../assets/modules/jquery.min.js"></script>
    <script src="../assets/modules/popper.js"></script>
    <script src="../assets/modules/tooltip.js"></script>
    <script src="../assets/modules/bootstrap/js/bootstrap.min.js"></script>
    <script src="../assets/modules/nicescroll/jquery.nicescroll.min.js"></script>
    <script src="../assets/modules/moment.min.js"></script>
    <script src="../assets/js/stisla.js"></script>

    <!-- JS Libraies -->
    <script src="../assets/modules/datatables/datatables.min.js"></script>
    <script src="../assets/modules/datatables/DataTables-1.10.16/js/dataTables.bootstrap4.min.js"></script>
    <script src="../assets/modules/datatables/Select-1.2.4/js/dataTables.select.min.js"></script>
    <script src="../assets/modules/jquery-ui/jquery-ui.min.js"></script>

    <!-- Page Specific JS File -->
    <script>
    $(document).ready(function() {
        // Initialize DataTable
        $('#medical-records-table').DataTable({
            "language": {
                "url": "../assets/js/dataTables.indonesian.json"
            }
        });
    });
    </script>

    <!-- Template JS File -->
    <script src="../assets/js/scripts.js"></script>
    <script src="../assets/js/custom.js"></script>
</body>

</html>